/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     l2scp.c                                                        */
/*                                                                          */
/* description:  compute L2 scalar products of the right hand side and      */
/*               the basis functions of the fe-space                        */
/*               sets Dirichlet boundary values                             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

/*--------------------------------------------------------------------------*/
/*  calculates the L2 scalar product of the function f with each basis      */
/*  function of the mesh belonging to the dof admin of fh; the scalar       */
/*  products are stored in fh;                                              */
/*                                                                          */
/*    fh->vec[j] =  (f, phi_j)_\Omega                                       */
/*                                                                          */
/*  where phi_j are the basis functions belonging to fh->fe_space->admin    */
/*                                                                          */
/*--------------------------------------------------------------------------*/

void L2scp_fct_bas(REAL (*f)(const REAL_D), const QUAD *quad, 
		   DOF_REAL_VEC *fh)
{
  FUNCNAME("L2scp_fct_bas");
  static WORKSPACE  ws = {0, nil};
  TRAVERSE_STACK    *stack;
  const EL_INFO     *el_info;
  MESH              *mesh = nil;
  const QUAD_FAST   *quad_fast;
  REAL              *det_p = nil, *wdetf_qp = nil, *f_vec = nil, **phi;
  REAL              val, det = 0.0;
  REAL_D            *x_p = nil, x;
  int               iq, j, n_phi, dim;
  const DOF         *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF         *dof;
  const PARAMETRIC  *parametric;

  TEST_EXIT(fh,"no DOF_REAL_VEC fh\n");
  if (!f) return;

  TEST_EXIT(fh->fe_space,"no fe_space in DOF_REAL_VEC %s\n", NAME(fh));

  GET_STRUCT(mesh, fh->fe_space);
  GET_DOF_VEC(f_vec, fh);

  dim     = mesh->dim;

  get_dof = fh->fe_space->bas_fcts->get_dof_indices;
  n_phi   = fh->fe_space->bas_fcts->n_bas_fcts;

  if (!quad)
    quad = get_quadrature(dim, 2*fh->fe_space->bas_fcts->degree-2);
  quad_fast = get_quad_fast(fh->fe_space->bas_fcts, quad, INIT_PHI);
  phi = quad_fast->phi;

  if ((parametric = mesh->parametric)) {
    REALLOC_WORKSPACE(&ws, quad->n_points*(2 + DIM_OF_WORLD)*sizeof(REAL));
    det_p    = (REAL *) ws.work;
    wdetf_qp = det_p+quad->n_points;
    x_p      = (REAL_D *) (wdetf_qp+quad->n_points);
  }
  else {
    REALLOC_WORKSPACE(&ws, quad->n_points*sizeof(REAL));
    wdetf_qp = (REAL *)ws.work;
  }

  stack = get_traverse_stack();
  el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_COORDS);

  while(el_info)
  {
    dof = get_dof(el_info->el, fh->fe_space->admin, nil);

    if (parametric) {
      (*parametric->init_element)(el_info, parametric);
      (*parametric->coord_to_world)(el_info, quad, 0, nil, x_p);
      (*parametric->det)(el_info, quad, 0, nil, det_p);

      for (iq = 0; iq < quad->n_points; iq++)
	wdetf_qp[iq] = quad->w[iq]*det_p[iq]*(*f)(x_p[iq]);
    }
    else {
      switch(dim) {
	  case 1:
	    det = el_det_1d(el_info);
	    break;
#if DIM_OF_WORLD > 1
	  case 2:
	    det = el_det_2d(el_info);
	    break;
#if DIM_OF_WORLD > 2
	  case 3:
	    det = el_det_3d(el_info);
#endif
#endif
      }

      for (iq = 0; iq < quad->n_points; iq++) {
	coord_to_world(el_info, quad->lambda[iq], x);
	wdetf_qp[iq] = quad->w[iq]*det*(*f)(x);
      }
    }

    for (j = 0; j < n_phi; j++)
    {
      for (val = iq = 0; iq < quad->n_points; iq++)
	val += phi[iq][j]*wdetf_qp[iq];
      f_vec[dof[j]] += val;
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}

void L2scp_fct_bas_d(const REAL *(*f)(const REAL_D, REAL_D),
		     const QUAD *quad, DOF_REAL_D_VEC *fh)
{
  FUNCNAME("L2scp_fct_bas_d");
  static WORKSPACE ws = {0, nil};
  TRAVERSE_STACK  *stack;
  const EL_INFO   *el_info;
  MESH            *mesh = nil;
  const QUAD_FAST *quad_fast;
  REAL_D          *x_p = nil, x;
  REAL_D          *wdetf_qp = nil, *f_vec = nil;
  REAL            *det_p = nil, val, det, **phi;
  int             iq, j, n, n_phi, dim;
  const DOF       *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF       *dof;
  const PARAMETRIC  *parametric;

  TEST_EXIT(fh,"no DOF_REAL_D_VEC fh\n");
  if (!f) return;

  TEST_EXIT(fh->fe_space,"no fe_space in DOF_REAL_D_VEC %s\n", NAME(fh));

  GET_STRUCT(mesh, fh->fe_space);
  GET_DOF_VEC(f_vec, fh);

  dim = mesh->dim;

  get_dof = fh->fe_space->bas_fcts->get_dof_indices;
  n_phi   = fh->fe_space->bas_fcts->n_bas_fcts;

  if (!quad)
    quad = get_quadrature(dim, 2*fh->fe_space->bas_fcts->degree-2);
  quad_fast = get_quad_fast(fh->fe_space->bas_fcts, quad, INIT_PHI);
  phi = quad_fast->phi;

  if ((parametric = mesh->parametric))
  {
    REALLOC_WORKSPACE(&ws, quad->n_points*(1 + 2*DIM_OF_WORLD)*sizeof(REAL));
    det_p    = (REAL *)ws.work;
    wdetf_qp = (REAL_D *) (det_p+quad->n_points);
    x_p      = wdetf_qp+quad->n_points;
  }
  else
  {
    REALLOC_WORKSPACE(&ws, quad->n_points*sizeof(REAL_D));
    wdetf_qp = (REAL_D *)ws.work;
  }

  stack = get_traverse_stack();
  el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_COORDS);
  while(el_info)
  {
    dof = get_dof(el_info->el, fh->fe_space->admin, nil);

    if (parametric) {
      (*parametric->init_element)(el_info, parametric);
      (*parametric->coord_to_world)(el_info, quad, 0, nil, x_p);
      (*parametric->det)(el_info, quad, 0, nil, det_p);

      for (iq = 0; iq < quad->n_points; iq++) {
	(*f)(x_p[iq],  wdetf_qp[iq]);
	for (n = 0; n < DIM_OF_WORLD; n++)
	  wdetf_qp[iq][n] *= quad->w[iq]*det_p[iq];
      }
    }
    else {
      det = el_det_0cd(el_info);

      for (iq = 0; iq < quad->n_points; iq++)
      {
	coord_to_world(el_info, quad->lambda[iq], x);
	(*f)(x,  wdetf_qp[iq]);
	for (n = 0; n < DIM_OF_WORLD; n++)
	  wdetf_qp[iq][n] *= quad->w[iq]*det;
      }
    }

    for (j = 0; j < n_phi; j++)
    {
      for (n = 0; n < DIM_OF_WORLD; n++)
      {
	for (val = iq = 0; iq < quad->n_points; iq++)
	  val += phi[iq][j]*wdetf_qp[iq][n];
	f_vec[dof[j]][n] += val;
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}

/*--------------------------------------------------------------------------*/
/*  sets dirichlet boundary values g() in the vectors fh and uh and fills   */
/*  the vector boundary with the boundary type of the corresponding DOF     */
/*--------------------------------------------------------------------------*/

void dirichlet_bound(REAL (*g)(const REAL_D), DOF_REAL_VEC *fh,
		     DOF_REAL_VEC *uh, DOF_SCHAR_VEC *bound)
{
  FUNCNAME("dirichlet_bound");
  static WORKSPACE ws = {0, nil};
  TRAVERSE_STACK  *stack;
  const EL_INFO   *el_info;
  MESH            *mesh = nil;
  const DOF       *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *) = nil;
  const DOF       *dof;
  DOF             *bound_dof;
  const S_CHAR    *(*get_bound)(const EL_INFO *, S_CHAR *) = nil;
  const S_CHAR    *el_bound;
  S_CHAR          *bound_vec;
  const REAL      *(*interpol)(const EL_INFO *, int, const int *,
			       REAL (*)(const REAL_D),
			       REAL (*f_loc)(const EL_INFO *el_info,
					     const REAL lambda[N_LAMBDA]),
			       REAL *) = nil;
  const REAL      *dirichlet;
  REAL            *f_vec = nil, *u_vec = nil;
  int             j, j_dof, n_phi = 0, n_dirichlet;

  if (!fh && !uh && !bound) return;

  if (fh && uh)
    TEST_EXIT(fh->fe_space == uh->fe_space,"fe_spaces in fh and uh differ\n");
  else if (!fh && uh)
  {
    fh = uh;
    uh = nil;
  }

  if (fh)
  {
    GET_STRUCT(mesh, fh->fe_space);
    GET_DOF_VEC(f_vec, fh);
    if (uh) {GET_DOF_VEC(u_vec, uh);}

    get_bound = fh->fe_space->bas_fcts->get_bound;
    get_dof   = fh->fe_space->bas_fcts->get_dof_indices;
    n_phi     = fh->fe_space->bas_fcts->n_bas_fcts;
    mesh      = fh->fe_space->mesh;
  }
  else if (bound)
  {
    get_bound = bound->fe_space->bas_fcts->get_bound;
    get_dof   = bound->fe_space->bas_fcts->get_dof_indices;
    n_phi     = bound->fe_space->bas_fcts->n_bas_fcts;
    mesh      = bound->fe_space->mesh;
    bound_vec = bound->vec;

    stack = get_traverse_stack();
    el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_BOUND);
    while(el_info)
    {
      dof      = get_dof(el_info->el, bound->fe_space->admin, nil);
      el_bound = get_bound(el_info, nil);

      for (j = 0; j < n_phi; j++)
	bound_vec[dof[j]] = el_bound[j];

      el_info = traverse_next(stack, el_info);
    }
    free_traverse_stack(stack);
    return;
  }

  if (bound)
  {
    TEST_EXIT(fh->fe_space == bound->fe_space,
      "fh->fe_space, bound->fe_space differ!\n");

    bound_vec = bound->vec;
    FOR_ALL_DOFS(bound->fe_space->admin, bound_vec[dof] = 0);
  }
  else
    bound_vec = nil;

  REALLOC_WORKSPACE(&ws, n_phi*sizeof(DOF));
  bound_dof = (DOF *)ws.work;

  if (g)
    interpol = fh->fe_space->bas_fcts->interpol;

  stack = get_traverse_stack();
  el_info = traverse_first(stack, mesh, -1, 
			   CALL_LEAF_EL|FILL_COORDS|FILL_BOUND);
  while(el_info)
  {
    dof      = get_dof(el_info->el, fh->fe_space->admin, nil);
    el_bound = get_bound(el_info, nil);
    n_dirichlet = 0;

    if (bound_vec)
    {
      for (j = 0; j < n_phi; j++)
      {
	if (el_bound[j] >= DIRICHLET  &&  bound_vec[dof[j]] <= INTERIOR)
	{
/*--- Dirichlet boundary for this DOF not set! Set it now -------------------*/
	  bound_dof[n_dirichlet++] = j;
	}
	bound_vec[dof[j]] = el_bound[j];
      }
    }
    else
    {
      for (j = 0; j < n_phi; j++)
      {
	if (el_bound[j] >= DIRICHLET)
	{
/*--- set Dirichlet value for this DOF ------------------------------------*/
	  bound_dof[n_dirichlet++] = j;
	}
      }
    }
      
    if (n_dirichlet && g)
    {
/*--- compute coefficients for all Dirichlet nodes ------------------------*/
      dirichlet = (*interpol)(el_info, n_dirichlet, bound_dof, g, nil, nil);
      
      if (u_vec)
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  u_vec[j_dof] = f_vec[j_dof] = dirichlet[j];
	}
      }
      else
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  f_vec[j_dof] = dirichlet[j];
	}
      }
    }
    else if (n_dirichlet)
    {
      if (u_vec)
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  u_vec[j_dof] = f_vec[j_dof] = 0.0;
	}
      }
      else
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  f_vec[j_dof] = 0.0;
	}
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}

/*--------------------------------------------------------------------------*/
/*  sets dirichlet boundary values g() in the vectors fh and uh and fills   */
/*  the vector boundary with the boundary type of the corresponding DOF     */
/*  for vector valued problems                                              */
/*--------------------------------------------------------------------------*/

void dirichlet_bound_d(const REAL *(*g)(const REAL_D, REAL_D),
		       DOF_REAL_D_VEC *fh, DOF_REAL_D_VEC *uh,
		       DOF_SCHAR_VEC *bound)
{
  FUNCNAME("dirichlet_bound_d");
  static WORKSPACE ws = {0, nil};
  TRAVERSE_STACK  *stack;
  const EL_INFO   *el_info;
  MESH            *mesh = nil;
  const DOF       *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *) = nil;
  const DOF       *dof;
  DOF             *bound_dof;
  const S_CHAR    *(*get_bound)(const EL_INFO *, S_CHAR *) = nil;
  const S_CHAR    *el_bound;
  S_CHAR          *bound_vec;
  const REAL_D    *(*interpol)(const EL_INFO *, int, const int *b_no,
			       const REAL *(*)(const REAL_D, REAL_D), 
			       const REAL *(*)(const EL_INFO *el_info,
					 const REAL lambda[N_LAMBDA],
					 REAL_D val),
			       REAL_D *) = nil;
  const REAL_D    *dirichlet;
  REAL_D          *f_vec = nil, *u_vec = nil;
  int             n, j, j_dof, n_phi = 0, n_dirichlet;

  if (!fh && !uh && !bound) return;

  if (fh && uh)
    TEST_EXIT(fh->fe_space == uh->fe_space,"fe_spaces in fh and uh differ\n");
  else if (!fh && uh)
  {
    fh = uh;
    uh = nil;
  }

  if (fh)
  {
    GET_STRUCT(mesh, fh->fe_space);
    GET_DOF_VEC(f_vec, fh);
    if (uh) {GET_DOF_VEC(u_vec, uh);}

    get_bound = fh->fe_space->bas_fcts->get_bound;
    get_dof   = fh->fe_space->bas_fcts->get_dof_indices;
    n_phi     = fh->fe_space->bas_fcts->n_bas_fcts;
    mesh      = fh->fe_space->mesh;
  }
  else if (bound)
  {
    get_bound = bound->fe_space->bas_fcts->get_bound;
    get_dof   = bound->fe_space->bas_fcts->get_dof_indices;
    n_phi     = bound->fe_space->bas_fcts->n_bas_fcts;
    mesh      = bound->fe_space->mesh;
    bound_vec = bound->vec;

    stack = get_traverse_stack();
    el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL|FILL_BOUND);
    while(el_info)
    {
      dof      = get_dof(el_info->el, bound->fe_space->admin, nil);
      el_bound = get_bound(el_info, nil);

      for (j = 0; j < n_phi; j++)
	bound_vec[dof[j]] = el_bound[j];

      el_info = traverse_next(stack, el_info);
    }
    free_traverse_stack(stack);
    return;
  }

  if (bound)
  {
    TEST_EXIT(fh->fe_space == bound->fe_space,
      "fh->fe_space, bound->fe_space differ!\n");

    bound_vec = bound->vec;
    FOR_ALL_DOFS(bound->fe_space->admin, bound_vec[dof] = 0);
  }
  else
    bound_vec = nil;

  REALLOC_WORKSPACE(&ws, n_phi*sizeof(DOF));
  bound_dof = (DOF *)ws.work;

  if (g)
    interpol = fh->fe_space->bas_fcts->interpol_d;

  stack = get_traverse_stack();
  el_info = traverse_first(stack, mesh, -1, 
			   CALL_LEAF_EL|FILL_COORDS|FILL_BOUND);
  while(el_info)
  {
    dof      = get_dof(el_info->el, fh->fe_space->admin, nil);
    el_bound = get_bound(el_info, nil);
    n_dirichlet = 0;

    if (bound_vec)
    {
      for (j = 0; j < n_phi; j++)
      {
	if (el_bound[j] >= DIRICHLET  &&  bound_vec[dof[j]] <= INTERIOR)
	{
/*--- Dirichlet boundary for this DOF not set! Set it now -------------------*/
	  bound_dof[n_dirichlet++] = j;
	}
	bound_vec[dof[j]] = el_bound[j];
      }
    }
    else
    {
      for (j = 0; j < n_phi; j++)
      {
	if (el_bound[j] >= DIRICHLET)
	{
/*--- set Dirichlet value for this DOF ------------------------------------*/
	  bound_dof[n_dirichlet++] = j;
	}
      }
    }
      
    if (n_dirichlet && g)
    {
/*--- compute coefficients for all Dirichlet nodes ------------------------*/
      dirichlet = (*interpol)(el_info, n_dirichlet, bound_dof, g, nil, nil);
      
      if (u_vec)
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  for (n = 0; n < DIM_OF_WORLD; n++)
	    u_vec[j_dof][n] = f_vec[j_dof][n] = dirichlet[j][n];
	}
      }
      else
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  for (n = 0; n < DIM_OF_WORLD; n++)
	    f_vec[j_dof][n] = dirichlet[j][n];
	}
      }
    }
    else if (n_dirichlet)
    {
      if (u_vec)
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  for (n = 0; n < DIM_OF_WORLD; n++)
	    u_vec[j_dof][n] = f_vec[j_dof][n] = 0.0;
	}
      }
      else
      {
	for (j = 0; j < n_dirichlet; j++)
	{
	  j_dof = dof[bound_dof[j]];
	  for (n = 0; n < DIM_OF_WORLD; n++)
	    f_vec[j_dof][n] = 0.0;
	}
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}
