/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */
 
#include <string>
#include "include/support.h"
#include "include/program.h"
#include "include/environment.h"
#include "include/paritycommand.h"
#include "include/variable.h"
#include "include/parser.h"

static gboolean unset_command (GNode *node)
{
	LWPiece *piece = (LWPiece *) node->data;
	
	if (piece == NULL) 
		return FALSE;
	
	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol != NULL)
		if (TRUE == symbol->isCommand()) {
			LWCommand *cmd = (LWCommand *) symbol;
			cmd->reset();
		}
	
	return FALSE;
}

static gboolean unset_variable (LWVariable *variable)
{
	variable->clear();
	
	return FALSE;
}

static GtkWidget *copy_piece_widget (LWPiece *piece)
{
	const GdkColor markcolor = {0, 0xcc00, 0xcc00, 0xcc00};

	guint size = piece->getRow()->getBoard()->getPieceSize();
	GtkWidget *widget = gtk_event_box_new();
	gtk_widget_set_size_request (widget, size, size);
	gtk_widget_modify_bg (widget, GTK_STATE_NORMAL, &markcolor);
	gtk_widget_show (widget);

	LWPixmap *pixmap = piece->getBackgroundPixmap ();

	if (pixmap != NULL) {
		GtkWidget *image = gtk_image_new_from_pixbuf (pixmap->getPixbuf());	
		gtk_container_add (GTK_CONTAINER (widget), image);
		gtk_widget_show (image);
	}
		
	return widget;
}

static GtkWidget *widget_tree (GNode *node)
{
	GtkWidget *vbox = gtk_vbox_new (FALSE, 0);

	if (node == NULL) return vbox;
	
	if (node->data != NULL)
		gtk_box_pack_start (GTK_BOX (vbox), copy_piece_widget ((LWPiece *) node->data), FALSE, FALSE, 0);
	
	GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);
	
	for (GNode *n = node->children; n != NULL; n=n->next)
		gtk_box_pack_start (GTK_BOX(hbox), widget_tree (n), TRUE, TRUE, 5);

	return vbox;
}

/*! \brief Destructor
*/	
LWProgramData::~LWProgramData ()
{
	LWParser::unrefData();
	
	if (tree_piece != NULL) {
		g_node_traverse (tree_piece, G_POST_ORDER, G_TRAVERSE_ALL,
							-1, (GNodeTraverseFunc) unset_command, NULL);
	
		g_node_destroy (tree_piece);
	}
	
	if (list_vars != NULL) {
		g_slist_foreach (list_vars, (GFunc) unset_variable, NULL);
		
		g_slist_free (list_vars);
	}
}

void LWProgramData::registerVariable (LWVariable *variable)
{
	if (NULL == g_slist_find (list_vars, variable))
		list_vars = g_slist_prepend (list_vars, (gpointer) variable);
}

void  LWProgramData::setTree (GNode *a_tree)
{
	tree_piece = a_tree;
}

GNode *LWProgramData::getTree ()
{
	return tree_piece;	
}

/*! \brief Debug syntax tree

	Draw the syntax tree in new window. Method only useful in analasis and testing
	of algorithms used for creating syntax tree.
*/
void LWProgramData::debugTree ()
{
	GtkWidget *w = widget_tree (tree_piece);
	
	GtkWidget *sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw), w);

	
	GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_add (GTK_CONTAINER (window), sw);
	gtk_widget_show_all (window);
	
	g_signal_connect (G_OBJECT (window), "delete_event", G_CALLBACK(gtk_widget_destroy), NULL);
}


//-------------------------------------------------

guint LWParser::ref_data = 0;
LWPiece *LWParser::begin_piece = NULL;

LWParser::LWParser (LWProgram *a_program): program (a_program), enable_debug(FALSE)
{
	if (begin_piece == NULL) {
		LWBoard *board = new LWBoard (LW_TYPE_PROGRAM);
		LWRow *row = new LWRow (board);
		board->addRow (row);
		begin_piece = new LWPiece (row);
		row->addPiece(begin_piece);
		begin_piece->setSymbol ("begin");
	}
}

LWParser::~LWParser ()
{
	if (ref_data == 0)
		delete program;
}

void LWParser::unrefData ()
{
	if (--ref_data == 0) {
		delete begin_piece->getRow()->getBoard();
		begin_piece = NULL;
	}
}

void LWParser::matchWith (LWCommandSegment*pcmd, LWCommand *cmd, LWPiece *piece)
{
	try {
		pcmd->matchWith(cmd);
	} catch (LWMessage *msg) {
		msg->setPiece(piece);
		throw msg;
	}
}

/* False - continue the loop */
void LWParser::compareParity (gboolean **r, 
							LWCommand *cmd_a, LWPiece *piece_a,
							LWCommand *cmd_b, LWPiece *piece_b)
{
	/* cmd_a = ")" , cmd_b = "("*/
	LWCommandSegment *scmd_a=NULL;
	LWCommandSegment *scmd_b=NULL;
	
	if (cmd_a != NULL)
		if (TRUE == cmd_a->isSegment())
			scmd_a = dynamic_cast <LWCommandSegment*> (cmd_a);

	if (cmd_b != NULL)
		if (TRUE == cmd_b->isSegment())
			scmd_b = dynamic_cast <LWCommandSegment*> (cmd_b);

	if ((scmd_a != NULL) && (scmd_b != NULL))
		if (FALSE == scmd_b->isAlreadyMatched())

			if (TRUE == scmd_a->matchPrevCondition(cmd_b) &&
				TRUE == scmd_b->matchNextCondition(cmd_a)) {
				
					matchWith (scmd_b, cmd_a, piece_b);
					matchWith (scmd_a, cmd_b, piece_a);
				
				**r = TRUE;
				return;
			}
			
	if (scmd_b != NULL) 
									
		if (FALSE == scmd_b->isAlreadyMatched())
			if (TRUE == scmd_b->matchNextCondition(cmd_a)) {
				**r = TRUE;
				return;
			}
				
	if (scmd_a != NULL)
		if (TRUE == scmd_a->matchPrevCondition(cmd_b)) {
			**r = FALSE;
			return;
		}
			
	*r = NULL;
}

gboolean LWParser::comparePriority (LWPiece *a, LWPiece *b) 
{
	const guint maxprio = 16;
	
	LWSymbol *sym_a = a->getSymbol();
	LWSymbol *sym_b = b->getSymbol();

	LWCommand *cmd_a = NULL;
	LWCommand *cmd_b = NULL;
	
	guint prio_a = maxprio;
	guint prio_b = maxprio;
	
	LWLink link = LW_LINK_RIGHT;
	
	if (sym_a != NULL)
		if (TRUE == sym_a->isCommand()) {
			cmd_a = (LWCommand *) sym_a;
			prio_a = cmd_a->getPriority();
		}

	if (sym_b != NULL)
		if (TRUE == sym_b->isCommand()) {
			cmd_b = (LWCommand *) sym_b;
			prio_b = cmd_b->getPriority();
			link = cmd_b->getLinkType();
		}

	gboolean bo;
	gboolean *r = &bo;
	
	if (prio_a > 1 && prio_b > 1) {
		compareParity (&r, cmd_a, a, cmd_b, b);

		if (r != NULL) 
			return bo;
	}
	
	switch (link) {
		case LW_LINK_LEFT:
			return (prio_a > prio_b) ? TRUE : FALSE;
		case LW_LINK_RIGHT:
			return (prio_a >= prio_b) ? TRUE : FALSE;
	}

	g_return_val_if_reached (FALSE);
}

/* \brief Change the symbol assigned to piece

	It changes the symbol assigned to piece

	\param piece The piece
	\param symbolname Name of the symbol, that should be assigned to piece
*/
void LWParser::toggleToSymbol (LWPiece *piece, const gchar *symbolname) 
{
	if (g_ascii_strcasecmp(piece->getSymbol()->getName(), symbolname)) {
		piece->setSymbol (symbolname);
	}
}

void LWParser::lexScan (LWPiece *prev, LWPiece *piece)
{
	if (prev == NULL) return;
	g_return_if_fail (piece != NULL);
	
	LWSymbol *psymbol = prev->getSymbol();
	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol == NULL)
		return;
	
	if (TRUE == symbol->isValue())
		return;

	if (!g_ascii_strcasecmp (symbol->getName(), "opp") || !g_ascii_strcasecmp (symbol->getName(), "sub")) {

		//digit or just icon without meaning
		if (psymbol == NULL) {
			toggleToSymbol (piece, "sub");
			return;
		}

		//other value (variable,world, etc)
		if (TRUE == psymbol->isValue()) {
			toggleToSymbol (piece, "sub");
			return;
		}
		
		// ) or ]
		if (TRUE == psymbol->isCommand())
			if (TRUE == ((LWCommand *) psymbol)->isSegment() &&
				1 < ((LWCommand *) psymbol)->getPriority() &&
				TRUE == ((LWCommand *) psymbol)->canBeSkiped()) {
					toggleToSymbol (piece, "sub");
					return;
				}
	}
	
	if (!g_ascii_strcasecmp (symbol->getName(), "sub")) {
		toggleToSymbol (piece, "opp");
	}
}

LWCommand *LWParser::getNodeCommand (GNode *node)
{
	g_return_val_if_fail (node != NULL, NULL);
	
	LWPiece *piece = (LWPiece *) node->data;
	g_return_val_if_fail (piece != NULL, NULL);
	
	LWSymbol *symbol = piece->getSymbol();
	g_return_val_if_fail (symbol != NULL, NULL);
	g_return_val_if_fail (TRUE == symbol->isCommand(), NULL);
	
	return (LWCommand *) symbol;
}

/* \brief Single iteration of creating syntax tree.
   
   Find proper place basing on previous element and insert
   piece into syntax tree. It produces syntax tree using
   so called operators method (at least is very similar)

   \param prev Previously inserted piece
   \param piece Piece to insert, which can contain the command
*/
GNode *LWParser::parseIteration (GNode *prev, LWPiece *piece)
{
GNode *node;

	if (TRUE == comparePriority (piece, (LWPiece *) prev->data)) 
		return g_node_append_data (prev, (gpointer) piece);

	g_return_val_if_fail (NULL != piece->getSymbol(), NULL);
	g_return_val_if_fail (TRUE == piece->getSymbol()->isCommand(), NULL);

	do {
	 	prev = prev->parent;
	} while (FALSE == comparePriority (piece, (LWPiece *) prev->data));
	
	LWCommand *cmd = (LWCommand *) piece->getSymbol();
	
	if (TRUE == cmd->canBeSkiped()) {
		if (TRUE == cmd->isSegment())
			((LWCommandSegment*) cmd)->matchWith(getNodeCommand(prev));
		cmd->reset();
		return prev;
	}

	/* Inserting closing part of multipart operator for example brace */
	if (TRUE == cmd->isSegment())
		if (TRUE == ((LWCommandSegment*) cmd)->matchPrevCondition(getNodeCommand(prev)))
			return g_node_append_data (prev->parent, (gpointer) piece);
		
	/* Insert piece between prev and last child of prev */
	node = g_node_last_child (prev); 
	g_node_unlink (node);
	GNode *t = g_node_append_data (prev, (gpointer) piece);
	g_node_append (t, node);
	return t;
}

/*! \brief Przetwarzanie wartości

	Ze zbioru klocków reprezuntujących cyfry, zmienne, obrazki
	tworzy obiekt wartości, który może być przekazany do polecenia
	\param node Węzeł drzewa z klockami
	\return Stworzona wartość
*/
LWValue *LWParser::LWParser::computeValue (LWProgramData *pd, GNode *node)
{
	LWValue *value;
	
	LWPiece *piece = (LWPiece *) node->data;
	g_return_val_if_fail (piece != NULL, NULL);

	LWSymbol *symbol = piece->getSymbol();
	
	if (symbol != NULL) {
		g_return_val_if_fail (TRUE == symbol->isValue(), NULL);
		
		value = dynamic_cast <LWValue *> (symbol);
		
		if (TRUE == value->isVariable()) {
			if (node->children == NULL) {
				pd->registerVariable ((LWVariable *) value);
				return value;
			} else {
				LWMessage *m = new LWMessage (LW_ERROR_BadString);
				m->setPiece (piece);
				throw m;
			}
		} else
			value = new LWValue (value);
		
	} else {
		LWPixmap *pixmap = piece->getBackgroundPixmap();
		value = new LWValue (pixmap);
	}
			
	for (GNode *n=node->children; n != NULL; n = n->children) {
		piece = (LWPiece *) n->data;
		g_return_val_if_fail (piece != NULL, NULL);

		symbol = piece->getSymbol();
		
		if (symbol != NULL) {
			g_return_val_if_fail (TRUE == symbol->isValue(), NULL);
			
			LWValue *v = dynamic_cast <LWValue *> (symbol);
			
			if (TRUE == v->isVariable()) {
				delete value;
				LWMessage *m = new LWMessage (LW_ERROR_BadString);
				m->setPiece (piece);
				throw m;
			}
			
			value->concat (v);
			continue;
		}
		
		LWPixmap *pixmap = piece->getBackgroundPixmap();
		value->append (pixmap);
	}
	
	return value;		
}

/*! \brief Set argument for a command

	It sets the argument for a command
	\param cmd The command
	\param node The node, which is will be used for computing value of the argument
*/
void LWParser::setArguments (LWProgramData *pd, LWCommand *cmd, GNode *node)
{
gint args=0;
GSList *destroy_list=NULL;
	
	for (GNode *n = node; n != NULL; n = n->next) {
		LWPiece *piece = (LWPiece *) n->data;
		g_return_if_fail (piece != NULL);
		
		LWSymbol *symbol = piece->getSymbol();
		LWCommand *cmd2 = NULL;
		
		if (symbol != NULL)
			if (TRUE == symbol->isCommand())
				cmd2 = (LWCommand *) symbol;
		
		if (cmd2 != NULL) {
			if (TRUE == cmd2->hasReturn())
				cmd->setArgument (args++, NULL);
		} else {
			cmd->setArgument (args++, computeValue (pd, n));
			destroy_list = g_slist_prepend (destroy_list, n);
		}
	}

	cmd->setArgc (args);	
	
	g_slist_foreach (destroy_list, (GFunc) g_node_destroy, NULL);
	g_slist_free (destroy_list);
}

/*! \brief Sprawdzanie poprawności stworzonego węzła
	
	Sprawdza poprawność poleceń, na które wskazuje węzeł.
	W przypadku wystąpień błędów, będą żucane wyjątki.
	Dodatkowo poleceniu zawartemu w węźle, przydzialane są 
	argumenty (te, które mogą być ustalone na etapie analizy
	programu, czyli nie będące rezulatatem wykonania innych poleceń).

	\param node Sprawdzany węzeł
*/
void LWParser::checkNode (LWProgramData *pd, GNode *node)
{
	for (GNode *n = node; n != NULL; n = n->next) {
		LWPiece *piece = (LWPiece *) n->data;
		g_return_if_fail (piece != NULL);

		if (piece != begin_piece) {
	
			LWSymbol *symbol = piece->getSymbol();

			if (symbol == NULL)
				continue;
			
			if (FALSE == symbol->isCommand())
				continue;
			
			LWCommand *cmd = (LWCommand *) symbol;

			try {
				if (TRUE == cmd->isSegment()) {
					LWCommandSegment *scmd = (LWCommandSegment *) cmd;
				
					if (FALSE == scmd->isAlreadyMatched())
						scmd->matchWith (NULL);
				}
					
				setArguments (pd, cmd, n->children);
			} catch (LWMessage *msg) {
				if (piece != begin_piece)
					msg->setPiece (piece);
				throw msg;			
			}
		}
			
		checkNode (pd, n->children);
	} 
	
}

gboolean LWParser::parseIterCmd (GNode **nodeptr, GNode **nodeptr2, LWPiece *piece)
{
	g_return_val_if_fail (nodeptr != NULL, TRUE);
	g_return_val_if_fail (nodeptr2 != NULL, TRUE);
	g_return_val_if_fail (piece != NULL, TRUE);
			
	LWSymbol *symbol = piece->getSymbol();

	if (symbol == NULL) 
		return FALSE;
			
	if (FALSE == symbol->isCommand()) 
		return FALSE;

	LWCommand *cmd_b = (LWCommand *) symbol;
	
	if (cmd_b->getPriority() != 1)
		return FALSE;

	gboolean handled = FALSE;
			
	if (cmd_b->isSegment()) {
		LWCommandSegment *scmd_b = (LWCommandSegment *) cmd_b;
			
		for (GNode *n = *nodeptr; !G_NODE_IS_ROOT (n); n = n->parent) 
		{
			if ((LWPiece *) n->data == begin_piece) {

				if (n->prev != NULL)
					n = n->prev;
				else
					continue;
			}
		
			LWCommand *cmd_a = getNodeCommand (n);
				
			if (cmd_a->isSegment() == FALSE)
				continue;
			
			LWCommandSegment *scmd_a = (LWCommandSegment *) cmd_a;
		
			if (TRUE == scmd_a->matchNextCondition(cmd_b) &&
				TRUE == scmd_b->matchPrevCondition(cmd_a)) {
				
					if (FALSE == scmd_b->matchWith (cmd_a))
						continue;
				
					if (FALSE == scmd_a->matchWith (cmd_b))
						continue;
				
					handled = TRUE;

					if (FALSE == cmd_b->canBeSkiped()) {
						*nodeptr2 = g_node_append_data (n->parent, (gpointer) piece);
						*nodeptr = n->parent;
					} else { 			
						cmd_b->reset();
						*nodeptr2 = n;
							
						//Remember, in that case arguments will be loaded to previous command, skip command in block
						*nodeptr = (TRUE == cmd_a->requiresLogicBlock()) ? n->next : n->parent;
					}
									
					break;
			}
		}
	}

	if (handled == FALSE) {

		if ((*nodeptr)->children != NULL) {
			// "back from: "if if", "while while", "if while", "while if" "for to if", "for to for to", etc 
			for (GNode *n = *nodeptr; !G_NODE_IS_ROOT (n); n = n->parent) 
			{
				if ((LWPiece *) n->data == begin_piece)
					*nodeptr = n->parent;
				else
					break;
			}
		}
					
		if (FALSE == cmd_b->canBeSkiped())
			*nodeptr2 = g_node_append_data (*nodeptr, (gpointer) piece);
		else {
//			if (cmd_b->isSegment())
//				((LWCommandSegment *) cmd_b)->matchWith (NULL);

			*nodeptr2 = *nodeptr;
			
		}

		if (FALSE == cmd_b->requiresLogicBlock() && TRUE == cmd_b->isSegment())
			*nodeptr = *nodeptr2;
						
	}
			
	// "if" "else" "while" "to" "downto"
	if (TRUE == cmd_b->requiresLogicBlock ()  ) {
		*nodeptr = g_node_append_data (*nodeptr, (gpointer) begin_piece);
		return TRUE;
	}
			
	
	return TRUE;
}

/*! \brief Analiza programu

	Przeprowadza analizę planszy z programem,
	tworząc drzewo składniowe. W razie błędów
	metoda tworzy okno z błędem i zaznacza ikonę
	w związku z którą wystąpił

	\param program Plansza z programem
	\return Prawda jęśli analiza zakończyła się sukcesem
	w przeciwnym wypadku fałsz.
*/
LWProgramData *LWParser::parse (LWBoard *program_board)
{
	g_return_val_if_fail (program_board != NULL, NULL);	
	
	LWProgramData *pd = new LWProgramData();
	
	ref_data++;

	/* Pobranie listy klocków z poleceniami*/
	GSList *list_piece = program_board->getPieceList();
		
	GNode *node = g_node_new ((gpointer) begin_piece);
	GNode *node2 = node;
		
	LWPiece *prev = NULL;
	
	for (GSList *l = list_piece; l != NULL; l = l->next) {

		lexScan (prev, (LWPiece *) l->data);
		prev = (LWPiece *) l->data;
		
		try {
			if (FALSE == parseIterCmd (&node, &node2, (LWPiece *) l->data))
				node2 = parseIteration (node2, (LWPiece *) l->data);
		
		} catch (LWMessage *msg) {
			if (enable_debug == TRUE) {
				pd->setTree (g_node_get_root (node));
				pd->debugTree ();
			}
		
			msg->setPiece ( (LWPiece *) l->data);
			program->showMessage(msg);
			g_slist_free (list_piece);
			delete pd;
			return NULL;
		}
		
	}

	g_slist_free (list_piece);
	
	pd->setTree(g_node_get_root (node));
	
	if (enable_debug) {
		pd->debugTree ();
	}
		
	try {
		checkNode (pd, pd->getTree());
	} catch (LWMessage *msg) {
		program->showMessage(msg);
		delete pd;
		return NULL;
	}

#if 0
		pd->debugTree ();
#endif
	
return pd;
}

void LWParser::enableDebug (gboolean enable)
{
	enable_debug = enable;
}
