/*
 * LaTeXDrawPSTricksParserActions.java
 */
package latexDraw.parsers.pstricks;

import java.awt.geom.Arc2D;
import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.*;
import latexDraw.figures.properties.Arrowable;
import latexDraw.parsers.InvalidFormatCommandException;
import latexDraw.parsers.NotFullyManagedException;
import latexDraw.parsers.pstricks.PSTricksParameters.PositionParameters;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines actions to do on a given command for the LaTeXDraw project.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 
 * 09/06/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.5<br>
 */
public class LaTeXDrawPSTricksParserActions extends PSTricksParserActionsManager
{
	/** The parsed figures. */
	protected Vector<Figure> figures;
	
	/** The figures contained in the pscustom command. Cleared at the end of each <code>pscustom</code> command.*/
	public Vector<Figure> psCustFigures;
	

	
	/**
	 * The constructor by default.
	 */
	public LaTeXDrawPSTricksParserActions()
	{
		super();
		psCustFigures 	= new Vector<Figure>();
		figures 		= new Vector<Figure>();
	}
	
	
	

	@Override
	public void actionsOnText(Vector<PSTricksParameters> params) throws InvalidFormatCommandException
	{
		if(params.isEmpty()) return;
		
		if(params.lastElement().textParsed.length()!=0)
		{
			PSTricksParameters last = params.lastElement();
			
			PositionParameters pp = null;
			int i = params.size()-1;
			
			while(pp==null && i>=0)
			{
				pp = params.elementAt(i).getPosition();
				i--;
			}
			
			if(pp==null)
				pp = new PositionParameters();
			
			Text t = new Text(true);
			Point2D.Double position = moveCoordinate(params, new Point2D.Double(0,0));
			double angle = Math.toRadians(PSTricksParameters.getWholeRotationAngle(params));
			
			if(position==null) throw new InvalidFormatCommandException("text", -1); //$NON-NLS-1$
			
			if(last.textSize!=null && last.textSize.length()!=0)
				t.setSizeByCommand(last.textSize);
			
			if((last.textSeries!=null && last.textSeries.equals("b")) || last.textBold) //$NON-NLS-1$
				 t.setIsBold(true);
			else t.setIsBold(false);
			
			if((last.textShape!=null && (last.textShape.equals("sl") || //$NON-NLS-1$
			   last.textShape.equals("it")) || last.textItalic)) //$NON-NLS-1$
				 t.setIsItalic(true);
			else t.setIsItalic(false);
			
			t.setRotationAngle(-angle);
			t.setText(last.textParsed);
			t.setLinesColor(last.textColor);
			
			if(last.textFamily!=null && last.textFamily.length()!=0)
				t.setTextFontByFamily(last.textFamily);
			
			t.updateFontsBorders();
			
			LaTeXDrawPoint2D pos = new LaTeXDrawPoint2D(position.x*Figure.PPC-t.getWidth()/2., 
					(position.y*Figure.PPC-t.getFontMetrics().getHeight()/4.)*-1);
			
			if(pp.refPoint!=PositionParameters.REF_POINT_DEFAULT_CENTER)
			{
				double addX=0, addY=0;
				Point2D.Double gc = new Point2D.Double((pos.x+t.getWidth())/2.,(pos.y+t.getHeight())/2.);
				
				if(pp.refPoint==PositionParameters.REF_POINT_BASE)
					addY=-t.getFontMetrics().getDescent();
				else if(pp.refPoint==PositionParameters.REF_POINT_TOP)
					addY=t.getFontMetrics().getAscent()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_BOTTOM)
					addY=-t.getFontMetrics().getDescent()*2;
				else if(pp.refPoint==PositionParameters.REF_POINT_LEFT)
					addX=t.getWidth()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT)
					addX=-t.getWidth()/2.;
				else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_BASE)
				{
					addY=-t.getFontMetrics().getDescent();
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_BOTTOM)
				{
					addY=-t.getFontMetrics().getDescent()*2;
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_RIGHT+PositionParameters.REF_POINT_TOP)
				{
					addY=t.getFontMetrics().getAscent()/2.;
					addX=-t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BASE)
				{
					addY=-t.getFontMetrics().getDescent();
					addX=t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BOTTOM)
				{
					addY=-t.getFontMetrics().getDescent()*2;
					addX=t.getWidth()/2.;
				}else if(pp.refPoint==PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_TOP)
				{
					addY=t.getFontMetrics().getAscent()/2.;
					addX=t.getWidth()/2.;
				}
				
				addX+=pp.labelSep*Figure.PPC;
				
				Point2D.Double pt = rotatePoint(new Point2D.Double(gc.x+addX, gc.y-addY), gc, angle);
				pos.setLocation(pos.x+(pt.x-gc.x), pos.y-(pt.y-gc.y));
			}
			
			t.setPosition(pos);
			t.updateFontsBorders();
			
			if(last.psCustomP.fromPsCustom)
				psCustFigures.add(t);
			else
				figures.add(t);
		}
	}




	/**
	 * @return the figures
	 */
	public Vector<Figure> getFigures()
	{
		return figures;
	}


	
	/**
	 * Allows to set the parameters of a figure <code>f</code>.
	 */
	public void setFigureParameters(PSTricksParameters p, Figure f, boolean hasStar)
	{
		if(f==null) return ;
		
		if(f.canHaveShadow())
		{
			f.setHasShadow(p.isShadow);
			f.setShadowAngle(Math.toRadians(p.shadowAngle));
			f.setShadowSize(Math.abs(p.shadowSize)*Figure.PPC);
			f.setShadowColor(p.shadowCol);
		}
		
		if(f.canBeHatched())
		{
			f.setHatchingSep(p.hatchSep*Figure.PPC);
			f.setGradientAngle(Math.toRadians(p.gradAngle));
			f.setGradientEndColor(p.gradEnd);
			f.setGradientMidPoint(p.gradMidPoint);
			f.setGradientStartColor(p.gradBegin);
		}
	
		f.setLinesColor(p.lineColor);
		f.setBordersPosition(p.borderPos);
		f.setDoubleColor(p.dbleColor);
		f.setDoubleSep(Math.abs(p.dbleSep*Figure.PPC));
		f.setHasDoubleBoundary(p.dbleLine);
		f.setHatchingColor(p.hatchCol);
		if(p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_SOLID))
			 f.setHatchingStyle(PSTricksConstants.TOKEN_FILL_NONE);
		else f.setHatchingStyle(
				p.fillStyle.endsWith("*") ? p.fillStyle.substring(0, p.fillStyle.length()-1) : p.fillStyle);//$NON-NLS-1$
		f.setHatchingWidth((float)Math.abs(p.hatchWidth*Figure.PPC));
		f.setDotSep((float)  p.dotStep*Figure.PPC);
		f.setHatchingAngle(Math.toRadians(p.hatchAngle));
		f.setBlackDashLength((float)p.dashBlack*Figure.PPC);
		f.setWhiteDashLength((float)p.dashWhite*Figure.PPC);
		f.setInteriorColor(p.fillColor);
		f.setIsFilled(p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_SOLID) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH_F) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_HLINES_F) ||
				p.fillStyle.equals(PSTricksConstants.TOKEN_FILL_VLINES_F));
		f.setLineStyle(p.lineStyle);
		f.setThickness((float)(Math.abs(p.lineWidth*Figure.PPC)));
		
		if(hasStar)
		{
			f.setIsFilled(true);
			f.setInteriorColor(f.getLinesColor());
			f.setHasDoubleBoundary(false);	
			f.setLineStyle(PSTricksConstants.LINE_NONE_STYLE);
			f.setHatchingStyle(PSTricksConstants.TOKEN_FILL_NONE);
			f.setBordersPosition(PSTricksConstants.BORDERS_INSIDE);
		}
	}
	
	


	
	
	@Override
	public int actionsOnFrameEllipse(int line, boolean hasStar, Vector<PSTricksParameters> ps, String params, 
				boolean isFrame) throws InvalidFormatCommandException
	{
		try
		{
			if(ps.isEmpty()) return ERR_END_CMD_INDEX;
			
			Point2D.Double p3;
			PSTricksParameters p = ps.lastElement();
			Point2D.Double p1 = new Point2D.Double(p.origin.x, p.origin.y);
			Point2D.Double p2 = new Point2D.Double();
			PSTricksParameters newP = new PSTricksParameters(p);
			double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
			int end = parseParametersFrameEllipse(line, params, p1, p2, newP);
			double xunit = newP.unit==PSTricksConstants.DEFAULT_UNIT?  newP.xUnit : newP.unit;
			double yunit = newP.unit==PSTricksConstants.DEFAULT_UNIT? newP.yUnit : newP.unit ;
			Figure f;
			p1.setLocation(p1.x+p.origin.x, p1.y+p.origin.y);
			
			if(isFrame)
			{
				p2.setLocation(p2.x+p.origin.x, p2.y+p.origin.y);
				if(ps.size()>1)
				{
					p1 = moveCoordinate(ps, p1);
					p2 = moveCoordinate(ps, p2);
					p3 = new Point2D.Double((p1.x+p2.x)/2.,(p1.y+p2.y)/2.);
					p1 = rotatePoint(p1, p3, angle);
					p2 = rotatePoint(p2, p3, angle);
				}
				
				p1.setLocation(p1.x*Figure.PPC*xunit, p1.y*Figure.PPC*-1*yunit);
				p2.setLocation(p2.x*Figure.PPC*xunit, p2.y*Figure.PPC*-1*yunit);
				
				LaTeXDrawPoint2D pt1 = new LaTeXDrawPoint2D(p1.x, p2.y);
				LaTeXDrawPoint2D pt2 = new LaTeXDrawPoint2D(p2.x, p2.y);
				LaTeXDrawPoint2D pt3 = new LaTeXDrawPoint2D(p1.x, p1.y);
				LaTeXDrawPoint2D pt4 = new LaTeXDrawPoint2D(p2.x, p1.y);
				
				f = new LaTeXDrawRectangle(pt1, pt2, pt3, pt4, true);
				((LaTeXDrawRectangle)f).setIsRound(newP.frameArc>0);
				if(newP.frameArc>0) ((LaTeXDrawRectangle)f).setFrameArc(newP.frameArc);
			}
			else 
			{
				p1 = moveCoordinate(ps, p1);
				p1.setLocation(p1.x*Figure.PPC*xunit, p1.y*Figure.PPC*-1*yunit);
				p2.setLocation(p2.x*Figure.PPC*xunit, p2.y*Figure.PPC*-1*yunit);
				
				LaTeXDrawPoint2D pt1 = new LaTeXDrawPoint2D(p1.x-p2.x, p1.y+p2.y);
				LaTeXDrawPoint2D pt2 = new LaTeXDrawPoint2D(p1.x+p2.x, p1.y+p2.y);
				LaTeXDrawPoint2D pt3 = new LaTeXDrawPoint2D(p1.x-p2.x, p1.y-p2.y);
				LaTeXDrawPoint2D pt4 = new LaTeXDrawPoint2D(p1.x+p2.x, p1.y-p2.y);
				
				f = new Ellipse(pt1, pt2, pt3, pt4, true);
			}
			
			setFigureParameters(newP, f, hasStar);
			f.setRotationAngle(angle);
			
			if(p.psCustomP.fromPsCustom)
				psCustFigures.add(f);
			else
				figures.add(f);
			
			return end;
			
		}catch(Exception e) 
		{ 
			if(isFrame)
				throw new InvalidFormatCommandException("psframe", line);  //$NON-NLS-1$
			throw new InvalidFormatCommandException("psellipse", line);  //$NON-NLS-1$
		}
	}




	@Override
	public int actionsOnCircle(int line, boolean hasStar, Vector<PSTricksParameters> ps, String params) 
				throws InvalidFormatCommandException
	{
		try
		{
			PSTricksParameters p = ps.lastElement();
			Point2D.Double center = new Point2D.Double(p.origin.x, p.origin.y);
			double[] radius = {1};
			PSTricksParameters newP = new PSTricksParameters(p);
			Figure c;
			double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
			int end = parseParametersCircle(line, params, center, radius, newP);
			LaTeXDrawPoint2D pt1, pt2, pt3, pt4;
			center = moveCoordinate(ps, center);
			
			pt1 = new LaTeXDrawPoint2D((p.origin.x+center.x-radius[0])*Figure.PPC*newP.unit, 
										(p.origin.y+center.y+radius[0])*Figure.PPC*-1*newP.unit);
			pt2 = new LaTeXDrawPoint2D((p.origin.x+center.x+radius[0])*Figure.PPC*newP.unit, 
										(p.origin.y+center.y+radius[0])*Figure.PPC*-1*newP.unit);
			pt3 = new LaTeXDrawPoint2D((p.origin.x+center.x-radius[0])*Figure.PPC*newP.unit, 
										(p.origin.y+center.y-radius[0])*Figure.PPC*-1*newP.unit);
			pt4 = new LaTeXDrawPoint2D((p.origin.x+center.x+radius[0])*Figure.PPC*newP.unit, 
										(p.origin.y+center.y-radius[0])*Figure.PPC*-1*newP.unit);
			
			c = new Circle(pt1, pt2, pt3, pt4, true);
			setFigureParameters(newP, c, hasStar);
			c.setRotationAngle(angle);
			
			if(p.psCustomP.fromPsCustom)
				psCustFigures.add(c);
			else
				figures.add(c);
			
			return end;
			
		}catch(Exception e) { throw new InvalidFormatCommandException("pscircle", line); } //$NON-NLS-1$
	}



	
	

	@Override
	public int actionsOnDot(int line, Vector<PSTricksParameters> ps, String params,
			boolean noCoordinatePossible) throws InvalidFormatCommandException
	{
		try
		{
			if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
			// It can have several points defined
			Vector<Point2D.Double> center = new Vector<Point2D.Double>();
			PSTricksParameters last = ps.lastElement();
			PSTricksParameters newP = new PSTricksParameters(last);
			double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
			int end = parseParametersDot(line, params, center, newP, noCoordinatePossible);
			double xunit = newP.unit==PSTricksConstants.DEFAULT_UNIT?  newP.xUnit : newP.unit;
			double yunit = newP.unit==PSTricksConstants.DEFAULT_UNIT? newP.yUnit : newP.unit ;
			newP.arrowDotSD = newP.arrowDotSD+newP.arrowDotSN<0 ? Math.abs(newP.arrowDotSD) : newP.arrowDotSD;
			newP.arrowDotSN = newP.arrowDotSD+newP.arrowDotSN<0 ? Math.abs(newP.arrowDotSN) : newP.arrowDotSN;
			newP.dotScale1 = Math.abs(newP.dotScale1);
				
			for(Point2D.Double pt: center)
			{
				pt = moveCoordinate(ps, pt);
				Dot d = new Dot(new LaTeXDrawPoint2D(pt.x*Figure.PPC*xunit, pt.y*Figure.PPC*-1*yunit), 
									true);
				setFigureParameters(newP, d, false);
				
				d.setCurrentStyle(newP.dotStyle);
				d.setWidth((float)((newP.arrowDotSD+newP.arrowDotSN*newP.lineWidth)*newP.dotScale1)*Figure.PPC);
				d.setRotationAngle(Math.toRadians(-newP.dotAngle)+angle);
				d.setInteriorColor(newP.fillColor);
				
				if(!last.psCustomP.fromPsCustom)
					figures.add(d);
			}
			
			return end;
			
		}catch(Exception e) { throw new InvalidFormatCommandException("psdot", line); }//$NON-NLS-1$
	}


	
	


	@Override
	public int actionsOnLine(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String params) throws InvalidFormatCommandException
	{
		try
		{
			if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
			
			Vector<Point2D.Double> pts = new Vector<Point2D.Double>();
			PSTricksParameters last = ps.lastElement();
			PSTricksParameters newP = new PSTricksParameters(last);
			Figure f;
			int end = parseParametersLine(line, params, pts, newP);
			double xunit = newP.unit==PSTricksConstants.DEFAULT_UNIT?  newP.xUnit : newP.unit;
			double yunit = newP.unit==PSTricksConstants.DEFAULT_UNIT? newP.yUnit : newP.unit ;
			
			if(pts.isEmpty()) return end;
			
			if(ps.size()>1)
				for(Point2D.Double pt : pts)
					pt.setLocation(moveCoordinate(ps, pt));
			
			if(pts.size()>2)//joined lines
				f = new JoinedLines(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), true);
			else
				f = new Line(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), true);
				
			
			Point2D.Double ptOld= null;
			
			for(Point2D.Double pt : pts)
				if(!pt.equals(ptOld))
				{
					((LaTeXDrawPolygon)f).addPoint(new LaTeXDrawPoint2D(pt.x*Figure.PPC*xunit,
													pt.y*Figure.PPC*-1*yunit));
					ptOld = pt;
				}

			if(pts.size()>2)//joined lines
			{
				((JoinedLines)f).removePointAt(0);
				((JoinedLines)f).removePointAt(0);
				((JoinedLines)f).setArrow2Style(ArrowHead.invertArrowStyle(newP.arrowStyle[1]));
				
				if(((JoinedLines)f).getNbPoints()<2)
					return end;
			}
			else
			{
				((Line)f).removePointAt(0);
				((Line)f).removePointAt(0);
				((Line)f).setArrow2Style(newP.arrowStyle[1]);
			}
			
			((Arrowable)f).setArrow1Style(newP.arrowStyle[0]);
			setArrowParameters(((Arrowable)f).getArrowHead1(), newP);
			setArrowParameters(((Arrowable)f).getArrowHead2(), newP);
			setFigureParameters(newP, f, hasStar);
			
			if(last.psCustomP.fromPsCustom)
				psCustFigures.add(f);
			else
				figures.add(f);
			
			return end;
			
		}catch(Exception e) { e.printStackTrace(); throw  new InvalidFormatCommandException("psline", line); }//$NON-NLS-1$
	}

	
	
	
	/**
	 * Sets the parameters of an arrowhead.
	 * @param ah The arrowhead.
	 * @param param The parameters.
	 * @since 0.2.1
	 */
	protected void setArrowParameters(ArrowHead ah, PSTricksParameters param)
	{
		ah.setArrowInset(param.arrowInset);
		ah.setArrowLength(param.arrowLgth);
		ah.setArrowSizeDim(param.arrowSizeD*Figure.PPC);
		ah.setArrowSizeNum(param.arrowSizeN);
		ah.setBracketNum(param.arrowBrLgth);
		ah.setDotSizeDim((param.arrowDotSD+param.arrowDotSN<0 ? Math.abs(param.arrowDotSD) : param.arrowDotSD)*Figure.PPC);
		ah.setDotSizeNum(param.arrowDotSD+param.arrowDotSN<0 ? Math.abs(param.arrowDotSN) : param.arrowDotSN);
		ah.setRBracketNum(param.arrowrBrLgth);
		ah.setTBarSizeDim(param.arrowTBarSD+param.arrowTBarSD<0 ? Math.abs(param.arrowTBarSD) : param.arrowTBarSD*Figure.PPC);
		ah.setTBarSizeNum(param.arrowTBarSD+param.arrowTBarSD<0 ? Math.abs(param.arrowTBarSN) : param.arrowTBarSN);

	}



	@Override
	public int actionsOnPolygon(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String params) throws InvalidFormatCommandException
	{
		try
		{
			if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
			
			Vector<Point2D.Double> pts = new Vector<Point2D.Double>();
			PSTricksParameters last = ps.lastElement();
			PSTricksParameters newP = new PSTricksParameters(last);
			int end = parseParametersPolygon(line, params, pts, newP);
			double xunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.xUnit;
			double yunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.yUnit;

			if(pts.size()<2) return end;
			
			LaTeXDrawPolygon pol = new LaTeXDrawPolygon(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), true);
			Point2D.Double ptOld= null;
			
			if(pts.size()==2)
				pts.add(new Point2D.Double(newP.origin.x, newP.origin.y));
			
			for(Point2D.Double pt : pts)
			{
				pt.setLocation(moveCoordinate(ps, pt));
				if(!pt.equals(ptOld))
				{
					pol.addPoint(new LaTeXDrawPoint2D(pt.x*Figure.PPC*xunit, pt.y*Figure.PPC*-1*yunit));
					ptOld = pt;
				}
			}
			
			pol.removePointAt(0);
			pol.removePointAt(0);
			
			if(pol.getNbPoints()<3)
				return end;
			
			setFigureParameters(newP, pol, hasStar);
			
			pol.updateBorders();
			pol.updateGravityCenter();
			
			if(last.psCustomP.fromPsCustom)
				psCustFigures.add(pol);
			else
				figures.add(pol);
			
			return end;
			
		}catch(Exception e) { throw new InvalidFormatCommandException("pspolygon", line); }//$NON-NLS-1$
	}




	
	
	
	@Override
	public int actionsOnDiamondTriangle(int line, boolean hasStar, Vector<PSTricksParameters> ps,
			String params, boolean isDiamondNotTriangle) throws InvalidFormatCommandException
	{
		try
		{
			if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
			
			PSTricksParameters p = ps.lastElement();
			Point2D.Double p1 = new Point2D.Double(p.origin.x, p.origin.y);
			Point2D.Double p2 = new Point2D.Double();
			PSTricksParameters newP = new PSTricksParameters(p);
			Figure f;
			double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
			int end = parseParametersFrameEllipse(line, params, p1, p2, newP);
			double xunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.xUnit;
			double yunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.yUnit;
			p1.setLocation(p1.x+p.origin.x, p1.y+p.origin.y);
			double moduloA;
			if(ps.size()>1) moduloA = ps.elementAt(ps.size()-2).degrees;
			else moduloA = 360;

			if(isDiamondNotTriangle)
			{
				if(ps.size()>1)
					p1 = moveCoordinate(ps, p1);
				
				LaTeXDrawPoint2D pt1 = new LaTeXDrawPoint2D((p1.x-p2.x)*Figure.PPC*xunit, 
															(p1.y-p2.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt2 = new LaTeXDrawPoint2D((p1.x+p2.x)*Figure.PPC*xunit, 
															(p1.y-p2.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt3 = new LaTeXDrawPoint2D((p1.x-p2.x)*Figure.PPC*xunit, 
															(p1.y+p2.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt4 = new LaTeXDrawPoint2D((p1.x+p2.x)*Figure.PPC*xunit, 
															(p1.y+p2.y)*Figure.PPC*-1*yunit);
				
				f = new Rhombus(pt1, pt2, pt3, pt4, true);
			}
			else 
			{
				Point2D.Double p3 = new Point2D.Double();
				if(ps.size()>1)
				{
					p1 = moveCoordinate(ps, p1);
					p3 = new Point2D.Double(p1.x,p1.y-p2.y/2.);
					p1 = rotatePoint(p1, p3, -angle);
				}
			
				LaTeXDrawPoint2D pt1 = new LaTeXDrawPoint2D((p1.x-p2.x/2.)*Figure.PPC*xunit, 
															(p1.y+p2.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt2 = new LaTeXDrawPoint2D((p1.x+p2.x/2.)*Figure.PPC*xunit, 
															(p1.y+p2.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt3 = new LaTeXDrawPoint2D((p1.x-p2.x/2.)*Figure.PPC*xunit, 
															(p1.y)*Figure.PPC*-1*yunit);
				LaTeXDrawPoint2D pt4 = new LaTeXDrawPoint2D((p1.x+p2.x/2.)*Figure.PPC*xunit, 
															(p1.y)*Figure.PPC*-1*yunit);
			
				f = new Triangle(pt1, pt2, pt3, pt4, true);
				
				if((newP.gangle%moduloA)!=0)
				{
					LaTeXDrawPoint2D gc = f.getGravityCenter();
					LaTeXDrawPoint2D base = new LaTeXDrawPoint2D((pt1.x+pt2.x)/2., pt3.y);
					LaTeXDrawPoint2D newGc = Figure.rotatePoint(gc, base, Math.toRadians(-(360/moduloA)*newP.gangle));
					f.shift(gc, newGc);
				}
			}
			
			setFigureParameters(newP, f, hasStar);
			
			if(isDiamondNotTriangle)
			{
				Rhombus r = ((Rhombus)f);
				r.setShadowAngle(r.getShadowAngle()-Math.toRadians(newP.gangle%moduloA));
				r.setHatchingAngle(r.getHatchingAngle()-Math.toRadians(newP.gangle%moduloA));
				r.setGradientAngle(r.getGradientAngle()-Math.toRadians(newP.gangle%moduloA));
				
				if((newP.gangle%moduloA)!=0)
					r.setRotationAngle(Math.toRadians(-(360/moduloA)*newP.gangle)+angle);
				else r.setRotationAngle(angle);
			}
			else
			{
				Triangle t = ((Triangle)f);
				boolean reverse = t.getPoint(0).y>t.getPoint(2).y;
				
				t.setRotationAngle(angle); // fixes #1556340
				if(reverse)
				{
					t.setShadowAngle(t.getShadowAngle()+Math.PI);
					t.setGradientAngle(t.getGradientAngle()+Math.PI);
				}
			}
			
			if(p.psCustomP.fromPsCustom)
				psCustFigures.add(f);
			else
				figures.add(f);
			
			return end;
			
		}catch(Exception e) 
		{ 
			if(isDiamondNotTriangle)
				throw new InvalidFormatCommandException("psdiamond", line); //$NON-NLS-1$
			throw new InvalidFormatCommandException("triangle", line); //$NON-NLS-1$
		}
	}

	
	
	
	
	 @Override
	public int actionsOnBezierCurve(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String params) throws InvalidFormatCommandException
		{
			try
			{
				if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
				
				Vector<Point2D.Double> pts = new Vector<Point2D.Double>();
				PSTricksParameters last = ps.lastElement();
				PSTricksParameters newP = new PSTricksParameters(last);
				LaTeXDrawPoint2D p1;
				int ended = parseParametersBezierCurve(line, params, pts, newP), i, size, j;
				double xunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.xUnit;
				double yunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.yUnit;
			
				if(pts.size()<3) return ended;
				
				for(Point2D.Double pt : pts)
					pt.setLocation(moveCoordinate(ps, pt));
				
				if(pts.size()%3==0) 
					p1 = new LaTeXDrawPoint2D(newP.origin.x*Figure.PPC*xunit, newP.origin.y*-1*Figure.PPC*yunit);
				else 
				{
					p1 = new LaTeXDrawPoint2D(pts.firstElement().x*Figure.PPC*xunit, pts.firstElement().y*-1*Figure.PPC*yunit);
					pts.removeElementAt(0);
				}
			
				Point2D.Double pt2;
				BezierCurve f = new BezierCurve(p1, 
												new LaTeXDrawPoint2D(pts.elementAt(2).x*Figure.PPC*xunit, 
																	 pts.elementAt(2).y*-1*Figure.PPC*yunit), true);
				
				f.setXCoordFirstCtrl(pts.firstElement().x*Figure.PPC*xunit, 0);
				f.setYCoordFirstCtrl(pts.firstElement().y*-1*Figure.PPC*yunit, 0);
				f.setXCoordFirstCtrl(pts.elementAt(1).x*Figure.PPC*xunit, 1);
				f.setYCoordFirstCtrl(pts.elementAt(1).y*-1*Figure.PPC*yunit, 1);
				
				i = 4;
				j = 2;
				size = pts.size();
		
				while(i<size)
				{
					pt2 = pts.elementAt(i);
					++i;
					
					f.addPoint(new LaTeXDrawPoint2D(pts.elementAt(i).x*Figure.PPC*xunit, 
													pts.elementAt(i).y*-1*Figure.PPC*yunit));
					
					f.setXCoordFirstCtrl(pt2.x*Figure.PPC*xunit, j);
					f.setYCoordFirstCtrl(pt2.y*-1*Figure.PPC*yunit, j);
					j++;
					i+=2;
				}
				
				setFigureParameters(newP, f, hasStar);
				
				f.updateSecondControlPoints();
				f.updateShape();
				f.updateBorders();
				f.setShowPoints(newP.showPoints);
				f.setOpen(true);
				f.replaceLastPointByClosing();
				
				((Arrowable)f).setArrow1Style(newP.arrowStyle[0]);
				((Arrowable)f).setArrow2Style(ArrowHead.invertArrowStyle(newP.arrowStyle[1]));
				setArrowParameters(((Arrowable)f).getArrowHead1(), newP);
				setArrowParameters(((Arrowable)f).getArrowHead2(), newP);
				
				if(f.getNbPoints()>1)
					if(last.psCustomP.fromPsCustom)
						psCustFigures.add(f);
					else
						figures.add(f);
				
				return ended;
				
			}catch(Exception e) { throw new InvalidFormatCommandException("psbezier", line); } //$NON-NLS-1$
		}



	 

	@Override
	public int actionsOnArcWedge(int line, boolean hasStar, Vector<PSTricksParameters> ps, String params, 
			boolean isArc) throws InvalidFormatCommandException
	{
		try
		{
			PSTricksParameters p = ps.lastElement();
			Point2D.Double center = (Point2D.Double)p.origin.clone();
			PSTricksParameters newP = new PSTricksParameters(p);
			double[] angle1 = {0}, angle2 = {0}, radius = {0};
			double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
			
			int end = parseParametersArcWedge(line, params, radius, center, angle1, angle2, newP, isArc);
		
			center = moveCoordinate(ps, center);
			
			Arc f = new Arc(new LaTeXDrawPoint2D((center.x-radius[0])*Figure.PPC*newP.xUnit, 
					(center.y+radius[0])*Figure.PPC*-1*newP.xUnit), 
					new LaTeXDrawPoint2D((center.x+radius[0])*Figure.PPC*newP.xUnit, 
					(center.y-radius[0])*Figure.PPC*-1*newP.xUnit), isArc ? Arc2D.OPEN : Arc2D.PIE, true);
			
			f.setStartAngle(Math.toRadians(angle1[0]));
			f.setEndAngle(Math.toRadians(angle2[0]));
			f.setShowPoints(newP.showPoints);
			f.setArrow1Style(newP.arrowStyle[0]);
			
			String currentArrowRStyle = newP.arrowStyle[1];
			if(currentArrowRStyle.equals(PSTricksConstants.DLARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DRARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.DRARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DLARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.LARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.RARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.DLARROW_STYLE))
				currentArrowRStyle = PSTricksConstants.DRARROW_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.RRBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.LRBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.LSBRACKET_STYLE;
			else if(currentArrowRStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
				currentArrowRStyle = PSTricksConstants.RSBRACKET_STYLE;
			
			f.setArrow2Style(currentArrowRStyle);
			
			setArrowParameters(f.getArrowHead1(), newP);
			setArrowParameters(f.getArrowHead2(), newP);
			setFigureParameters(newP, f, hasStar);
			f.setRotationAngle(angle);
			
			if(p.psCustomP.fromPsCustom)
				psCustFigures.add(f);
			else
				figures.add(f);
			
			return end;
			
		}catch(Exception e) 
		{ 
			if(isArc)
				throw new InvalidFormatCommandException("psarc", line);  //$NON-NLS-1$
			throw new InvalidFormatCommandException("pswedge", line);  //$NON-NLS-1$
		}
	}



	
	@Override
	public int actionsOnGridAxes(int line, Vector<PSTricksParameters> ps, Point2D.Double pictureSWPt, 
			Point2D.Double pictureNEPt, String params, boolean isGrid) throws InvalidFormatCommandException
		{
			try
			{
				if(ps==null || ps.isEmpty()) return ERR_END_CMD_INDEX;
				
				Vector<Point2D.Double> pts = new Vector<Point2D.Double>();
				PSTricksParameters last = ps.lastElement();
				PSTricksParameters newP = new PSTricksParameters(last);
				int gridEndX, gridEndY, gridStartX, gridStartY, originX, originY;
				double angle = Math.toRadians(-PSTricksParameters.getWholeRotationAngle(ps));
				int end = parseParametersGridAxes(line, params, pts, newP);
				double xunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.xUnit;
				double yunit = newP.unit!=PSTricksConstants.DEFAULT_UNIT? newP.unit : newP.yUnit;
				newP.origin.setLocation(moveCoordinate(ps, newP.origin));
				boolean isGridXLabelReverse = false;
				boolean isGridYLabelReverse = false;
				
				if(pts.isEmpty())
				{
					if(pictureNEPt.x>=0)
						 gridEndX = (pictureNEPt.x-((int)pictureNEPt.x))>0.5 ? (int)pictureNEPt.x+1 : (int)pictureNEPt.x;
					else gridEndX = (((int)pictureNEPt.x)-pictureNEPt.x)>0.5 ? (int)pictureNEPt.x-1 : (int)pictureNEPt.x;
					if(pictureNEPt.y>=0)
						 gridEndY = (pictureNEPt.y-((int)pictureNEPt.y))>0.5 ? (int)pictureNEPt.y+1 : (int)pictureNEPt.y;
					else gridEndY = (((int)pictureNEPt.y)-pictureNEPt.y)>0.5 ? (int)pictureNEPt.y-1 : (int)pictureNEPt.y;
				
					if(pictureSWPt.x>=0)
						 gridStartX = originX = (pictureSWPt.x-((int)pictureSWPt.x))>0.5 ? (int)pictureSWPt.x+1 : (int)pictureSWPt.x;
					else gridStartX = originX = (((int)pictureSWPt.x)-pictureSWPt.x)>0.5 ? (int)pictureSWPt.x-1 : (int)pictureSWPt.x;
					
					if(pictureSWPt.y>=0)
						 gridStartY = originY = (pictureSWPt.y-((int)pictureSWPt.y))>0.5 ? (int)pictureSWPt.y+1 : (int)pictureSWPt.y;
					else gridStartY = originY = (((int)pictureSWPt.y)-pictureSWPt.y)>0.5 ? (int)pictureSWPt.y-1 : (int)pictureSWPt.y;
				}
				else
					if(pts.size()==1)
					{
						gridStartX = gridStartY = originX = originY = 0;
						gridEndX = (int)pts.firstElement().x;
						gridEndY = (int)pts.firstElement().y;
					}
					else
						if(pts.size()==2)
						{
							originX = gridStartX = (int)pts.firstElement().x;
							originY = gridStartY = (int)pts.firstElement().y;
							gridEndX = (int)pts.elementAt(1).x;
							gridEndY = (int)pts.elementAt(1).y;
						}
						else
						{
							originX = (int)pts.firstElement().x;
							originY = (int)pts.firstElement().y;
							gridStartX = (int)pts.elementAt(1).x;
							gridStartY = (int)pts.elementAt(1).y;
							gridEndX = (int)pts.elementAt(2).x;
							gridEndY = (int)pts.elementAt(2).y;
						}
				
				if(gridStartX>=gridEndX)
				{
					int tmp = gridEndX;
					gridEndX = gridStartX;
					gridStartX = tmp;
					isGridXLabelReverse = true;
				}
				
				if(gridStartY>=gridEndY)
				{
					int tmp = gridEndY;
					gridEndY = gridStartY;
					gridStartY = tmp;
					isGridYLabelReverse = true;
				}
				
				LaTeXDrawPoint2D position = new LaTeXDrawPoint2D(newP.origin.x*Figure.PPC,newP.origin.y*Figure.PPC*-1);
				GridShape f=null;
				
				if(angle!=0)
				{
					Point2D.Double tmp1 = new Point2D.Double(
							position.x+((gridEndX+gridStartX)/2.)*Figure.PPC*xunit, 
							position.y-((gridEndY+gridStartY)/2.)*Figure.PPC*yunit);
					Point2D.Double tmp = rotatePoint(tmp1, position, angle);
					position.setLocation(position.x+(tmp.x-tmp1.x), position.y+(tmp.y-tmp1.y));
				}
				
				if(isGrid)
				{
					f = new Grid(position, true);
					Grid g = (Grid)f;
					g.setUnit(xunit);
					g.setGridDots(newP.gridDots);
					g.setGridLabelsColor(newP.labelsGridCol);
					g.setGridLabelsSize((int)((newP.gridLabel*PSTricksConstants.CM_VAL_PT)/0.6));
					g.setGridWidth((float)Math.abs(newP.gridWidth*Figure.PPC));
					g.setSubGridColor(newP.subGridCol);
					g.setSubGridDiv(newP.subGridDiv);
					g.setSubGridDots(newP.subGridDots);
					g.setSubGridWidth((float)Math.abs(newP.subGridWidth*Figure.PPC));
					g.setOriginX(originX);
					g.setOriginY(originY);
					g.setLinesColor(newP.gridColor);
				}
				else
				{
					f = new Axe(position, true);
					Axe a = (Axe)f;
					a.setOriginX((int)newP.ox);
					a.setOriginY((int)newP.oy);
					a.setLabelsDisplayedToken(newP.labels);
					a.setTicksDisplayedToken(newP.ticks);
					a.setTicksStyleToken(newP.ticksStyle);
					a.setTicksSize(newP.ticksSize*Figure.PPC);
					a.setIncrementX(newP.dxIncrement);
					a.setIncrementY(newP.dyIncrement);
					a.setDistLabelsX(newP.dxLabelDist);
					a.setDistLabelsY(newP.dyLabelDist);
					a.setShowOrigin(newP.showOrigin);
					a.setAxesStyleToken(newP.axesStyle);
					setFigureParameters(newP, a, false);
					
					String arrowHead2Style = newP.arrowStyle[1];
					
					if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
						arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.DRARROW_STYLE))
						arrowHead2Style = PSTricksConstants.DLARROW_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.RARROW_STYLE))
						arrowHead2Style = PSTricksConstants.LARROW_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.LARROW_STYLE))
						arrowHead2Style = PSTricksConstants.RARROW_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
						arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.LRBRACKET_STYLE))
						arrowHead2Style = PSTricksConstants.RRBRACKET_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.RRBRACKET_STYLE))
						arrowHead2Style = PSTricksConstants.LRBRACKET_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE))
						arrowHead2Style = PSTricksConstants.LSBRACKET_STYLE;
					else if(arrowHead2Style.equals(PSTricksConstants.LSBRACKET_STYLE))
						arrowHead2Style = PSTricksConstants.RSBRACKET_STYLE;
					
					((Arrowable)a).setArrow1Style(newP.arrowStyle[0]);
					((Arrowable)a).setArrow2Style(arrowHead2Style);
					setArrowParameters(((Arrowable)a).getArrowHead1(), newP);
					setArrowParameters(((Arrowable)a).getArrowHead2(), newP);
				}
				
				f.setGridEndX(gridEndX);
				f.setGridEndY(gridEndY);
				f.setGridStartX(gridStartX);
				f.setGridStartY(gridStartY);
				f.updateFonts();
				f.updateBorders(null);
				f.updateBorders();
				f.setRotationAngle(angle);
				f.setXLabelSouth(!isGridYLabelReverse);
				f.setYLabelWest(!isGridXLabelReverse);
				
				if(!last.psCustomP.fromPsCustom)
					figures.add(f);
			
				return end;
				
			}catch(Exception e) { throw new InvalidFormatCommandException("psgrid or psaxes", line); } //$NON-NLS-1$
		}




	@Override
	public int actionsOnPicture(int line, Vector<PSTricksParameters> ps, 
								String params) throws InvalidFormatCommandException
	{
		try
		{
			String[] path = {""};//$NON-NLS-1$
			int end = parseParametersPicture(line, params, path);
//			LaTeXDrawPoint2D position = new LaTeXDrawPoint2D();
//			BatchConvertFrame bcf = new BatchConvertFrame();
//			Image image = bcf.getImage(new File(path[0]));
//			BufferedImage bufferedI = new BufferedImage(image.getWidth(null), 
//											image.getHeight(null), BufferedImage.TYPE_INT_RGB );
//			Graphics g = bufferedI.createGraphics();
//			g.drawImage(image,0,0,null);
//			g.dispose();
//			
//			Picture p = new Picture(true, position, bufferedI, path[0]);
//			
//			figures.add(p);
			
			return end;
			
		}catch(Exception e)
		{
			e.printStackTrace();
			throw new InvalidFormatCommandException("includegraphics", line); //$NON-NLS-1$
		}
	}




	@Override
	public int actionsOnFramedBox(int nLine, boolean hasStar, Vector<PSTricksParameters> ps, 
			String cmd, String params, int type) throws InvalidFormatCommandException
	{
		int i = ps.size()-1;
		PSTricksParameters pp = null;
		
		while(pp==null && i>=0)
			if(ps.elementAt(i).getPosition()!=null)
				pp = ps.elementAt(i);
			else i--;
			
		if(pp==null)
			pp = ps.firstElement();
		
		FramedBox fb = new FramedBox(pp.psBoxText);
		fb.setStart(pp.textForFrame.length());
		pp.psBox.add(fb);
		PSTricksParameters newP = new PSTricksParameters(ps.lastElement());
		int end = -1;
		int[] j = {0};
		try 
		{ 
			goToNextChar(params, j); 
			if(params.charAt(j[0])=='*')
			{
				j[0]++;
				goToNextChar(params, j); 
			}
		} 
		catch(Exception e1) { throw new InvalidFormatCommandException(nLine); }
		
		if(params.charAt(j[0])=='[')
		{
			switch(type)
			{
				case 0: 
					fb.setBoxType(FramedBox.BOX_RECTANGLE); 
					end = parseParametersFrameEllipse(nLine, params, null, null, newP);
					if(newP.frameArc>0) 
					{
						((LaTeXDrawRectangle)fb.getBox()).setFrameArc(newP.frameArc);
						((LaTeXDrawRectangle)fb.getBox()).setIsRound(true);
					}
					break;
				case 1: 
					fb.setBoxType(FramedBox.BOX_CIRCLE); 
					end = parseParametersCircle(nLine, params, null, null, newP);
					break;
				case 2: 
					fb.setBoxType(FramedBox.BOX_TRIANGLE); 
					end = parseParametersPolygon(nLine, params, null, newP);
					break;
				case 3: 
					fb.setBoxType(FramedBox.BOX_DIAMOND); 
					end = parseParametersPolygon(nLine, params, null, newP);
					break;
				case 4: 
					fb.setBoxType(FramedBox.BOX_ELLIPSE);
					end = parseParametersFrameEllipse(nLine, params, null, null, newP);
					break;
				case 5: 
					fb.setBoxType(FramedBox.BOX_RECTANGLE); 
					end = parseParametersFrameEllipse(nLine, params, null, null, newP);
					if(newP.frameArc>0) 
					{
						((LaTeXDrawRectangle)fb.getBox()).setFrameArc(newP.frameArc);
						((LaTeXDrawRectangle)fb.getBox()).setIsRound(true);
					}
					break;
				case 6: 
					fb.setBoxType(FramedBox.BOX_RECTANGLE); 
					end = parseParametersFrameEllipse(nLine, params, null, null, newP);
					if(newP.frameArc>0) 
					{
						((LaTeXDrawRectangle)fb.getBox()).setFrameArc(newP.frameArc);
						((LaTeXDrawRectangle)fb.getBox()).setIsRound(true);
					}
					break;
			}
			j[0] = end;
		}
		
		try
		{
			goToNextChar(params, j);
		} catch(Exception e1) { throw new InvalidFormatCommandException(nLine); }
		
		switch(type)
		{
			case 0: 
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
			case 1: 
				fb.setBoxType(FramedBox.BOX_CIRCLE); 
				break;
			case 2: 
				fb.setBoxType(FramedBox.BOX_TRIANGLE); 
				break;
			case 3: 
				fb.setBoxType(FramedBox.BOX_DIAMOND); 
				break;
			case 4: 
				fb.setBoxType(FramedBox.BOX_ELLIPSE);
				break;
			case 5: 
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
			case 6: 
				fb.setBoxType(FramedBox.BOX_RECTANGLE); 
				break;
		}

		
		if(params.charAt(j[0])!='{')//psframebox3, psframebox[framesep=0.3]3 for instance.
		{
			PSTricksParameters p;
			boolean again = true;
			int k = ps.size()-1;
			
			while(again && k>=0)
			{
				p = ps.elementAt(k);
				if(p.getPosition()!=null)
				{
					again = false;
					p.textForFrame+=params.charAt(j[0]);
				}
				else 
				{
					p.textForFrame+=params.charAt(j[0]);
					k--;
				}
			}
			
			k = ps.size()-1;
			p=null;
			
			while(p==null && k>=0)
				if(ps.elementAt(k).getPosition()!=null)
					p = ps.elementAt(k);
				else k--;
			
			if(p==null)
				p = ps.firstElement();
			
			p.psBox.lastElement().setEnd(p.textForFrame.length());
			p.psBox.add(0, p.psBox.remove(p.psBox.size()-1));
			
			end=j[0]+1;
		}
		
		fb.setFrameSep(newP.frameSep*Figure.PPC);
		fb.setBoxSep(newP.boxSep);
		
		setFigureParameters(newP, fb.getBox(), hasStar);
		
		if(hasStar)
		{
			fb.getBox().setIsFilled(true);
			fb.getBox().setInteriorColor(newP.fillColor);
			fb.getBox().setLinesColor(newP.fillColor);
		}
		
		switch(type)
		{
			case 5: 
				fb.getBox().setHasDoubleBoundary(true);
				break;
			case 6: 
				fb.getBox().setHasShadow(true);
				break;
		}
		
		if(end==-1)
			end = j[0];
		
		return end;
	}




	
	@Override
	public void actionsOnTerminatedFramedBoxes(Vector<PSTricksParameters> params) throws InvalidFormatCommandException
	{
		PSTricksParameters pp = params.lastElement();
		
		String str = pp.psBoxText.getText();
		
		if(str.length()==0)
			str = " ";//$NON-NLS-1$
		
		pp.textParsed = pp.textForFrame;
		actionsOnText(params);
		
		if(pp.psBox.isEmpty())
			return;
		
		Text txt = (Text)figures.remove(figures.size()-1);
		boolean simpleBox = pp.psBox.size()==1 && pp.psBox.firstElement().getBoxedText().equals(txt.getText());
		
		if(simpleBox)
		{
			FramedBox fb = pp.psBox.firstElement();
			fb.setText(txt);
			fb.setStart(-1);
			fb.setEnd(-1);
			fb.getBox().setBordersPosition(PSTricksConstants.BORDERS_OUTSIDE);
			txt.setSimpleBox(fb);
		}
		else
			for(FramedBox fb : pp.psBox)
			{
				fb.setText(txt);
				txt.addBox(fb);
				fb.getBox().setBordersPosition(PSTricksConstants.BORDERS_OUTSIDE);
			}
		
		txt.setHasFramedBox(!pp.psBox.isEmpty());
		txt.setHasSimpleFramedBox(simpleBox);
		txt.updateFramedBoxes();
		txt.updateFontsBorders();
		
		if(!pp.psCustomP.fromPsCustom)
			figures.add(txt);
		
		if(simpleBox)
		{
			if(txt.getSimpleBox().getBoxType()==FramedBox.BOX_TRIANGLE)
			{
				double height = (txt.getSimpleBox().getBox().getTheSEBoundPoint().y-
						txt.getSimpleBox().getBox().getTheNWBoundPoint().y)/4.;
				LaTeXDrawPoint2D pos = txt.getPosition();
				txt.setPosition(new LaTeXDrawPoint2D(pos.x, pos.y+height));
				txt.getSimpleBox().getBox().updateShape();
			}
		}
		else
		{
			FramedBox fb, max=null;
			double xMin=Double.MAX_VALUE;
			int i=0, size = txt.getMultipleBox().size();
			
			while(i<size)
			{
				fb = txt.getMultipleBox().elementAt(i);

				if(fb.isBoxSep() && fb.getBoxType()==FramedBox.BOX_TRIANGLE &&
					fb.getBox().getTheNWBoundPoint().x<xMin)
				{
					xMin = fb.getBox().getTheNWBoundPoint().x;
					max = fb;
				}
				else i++;
			}

			if(max!=null)
			{
				double height = (max.getBox().getTheSEBoundPoint().y-max.getBox().getTheNWBoundPoint().y)/4.;
									
				LaTeXDrawPoint2D pos = txt.getPosition();
				txt.setPosition(new LaTeXDrawPoint2D(pos.x, pos.y+height));
				max.getBox().updateShape();
			}
			
		}
		
		txt.updateFramedBoxes();
		txt.updateFontsBorders();
	}


	
	
	
	/**
	 * Optimises the figures; for instance, frame are transformed in square.
	 * @param vFigures The figures to optimise.
	 * @since 1.9
	 */
	public static void optimizeCode(Vector<Figure> vFigures)
	{
		Vector<Arc> vArcs 	= new Vector<Arc>();
		Vector<Line> vLines = new Vector<Line>();
		Vector<BezierCurve> vCurves = new Vector<BezierCurve>();
		int sizevl, i, size, j;
		
		for(Figure f : vFigures)
			if(f instanceof Line)
				vLines.add((Line)f);
			else
				if(f instanceof Arc)
					vArcs.add((Arc)f);
				else
					if(f instanceof BezierCurve)
						vCurves.add((BezierCurve)f);

		sizevl = vLines.size();
		
		for(Arc a : vArcs)// We check if we can convert arc+line in chord.
			if(a.getType()==Arc2D.OPEN)
			{
				boolean again = true;
				i=0;
				LaTeXDrawPoint2D startP, endP, p1, p2;
				Line l=null;
				
				while(again && i<sizevl)
				{
					l		= vLines.elementAt(i);
					startP 	= a.getStartPoint(true);
					endP	= a.getEndPoint(true);
					p1		= l.getPt1();
					p2		= l.getPt2();
					startP.y*=-1;
					endP.y*=-1;
					
					if((startP.equals(p1, 1) && endP.equals(p2, 1)) || 
						(startP.equals(p2, 1) && endP.equals(p1, 1)))
						again = false;
					else 
						i++;
				}
				
				if(!again && a.isParametersEquals(l, false, true))
				{
					vFigures.remove(l);
					a.setType(Arc2D.CHORD);
				}
			}
		
		if(vCurves.size()>1)// We try to concat the Bézier curves.
		{
			if(vCurves.size()<200)// We must set a limit otherwise it may have a freeze.
			{
				BezierCurve b1, b2;
			
				for(j=0; j<vCurves.size(); j++)
				{
					i = j+1;
					b1 = vCurves.elementAt(j);
					
					while(i<vCurves.size())
					{
						b2 = vCurves.elementAt(i);
					
						if(b1.isJoined(b2)==1 && b1.isParametersEquals(b2, true, true))
						{
							vCurves.remove(b2);
							vFigures.remove(b2);
							b1.join(b2);
							i=j+1;
						}
						else
							i++;
					}
				} //for
			}
			
			for(BezierCurve bc : vCurves)
				bc.replaceLastPointByClosing();
		}//if
		
		i = 0;
		while(i<vCurves.size()) // We check if the it is possible to close a curve.
		{
			BezierCurve bc = vCurves.elementAt(i);
			bc.replaceLastPointByClosing();
			
			if(bc.isOpen())
			{
				boolean again = true;
				size = vLines.size();

				for(int k=0; k<size && again; k++)//We try to close it with a line.
					if(bc.isLineClosingCurve(vLines.elementAt(k)) && 
						bc.isParametersEquals(vLines.elementAt(k), true, false))
					{
						bc.setOpen(false);
						bc.setCloseType(BezierCurve.CLOSE_TYPE_LINE);
						again = false;
						vFigures.remove(vLines.elementAt(k));
					}
				
				if(again)//We try to close it with a two-points curve.
				{
					j = 0;
					
					while(j<i && again)
					{
						BezierCurve bc2 = vCurves.elementAt(j);
						
						if(bc2.getNbPoints()==2 && bc.isOpen() && bc2.isCurveClosingCurve(bc)==1 && 
							bc.isParametersEquals(bc2, true, false))
						{
							bc.setOpen(false);
							bc.setCloseType(BezierCurve.CLOSE_TYPE_CURVE);
							again = false;
							vCurves.remove(j);
							vFigures.remove(bc2);
							i--;
						}
						else j++;
					}
					
					j = i+1;
					while(j<vCurves.size() && again)
					{
						BezierCurve bc2 = vCurves.elementAt(j);
						
						if(bc2.getNbPoints()==2 && bc.isOpen() && bc.isCurveClosingCurve(bc2)==1 && 
							bc.isParametersEquals(bc2, true, false))
						{
							bc.setOpen(false);
							bc.setCloseType(BezierCurve.CLOSE_TYPE_CURVE);
							again = false;
							vCurves.remove(j);
							vFigures.remove(bc2);
						}
						else j++;
					}
				}
			}//if(bc.isOpen())
			i++;
		}
		
		i=0;
		size = vFigures.size();
		while(i<size)
		{
			Figure f = vFigures.elementAt(i);
			
			if(f instanceof LaTeXDrawRectangle && !(f instanceof Square))
			{// We check if we can convert frame in square.
				LaTeXDrawRectangle frame = (LaTeXDrawRectangle)f;
				
				if(frame.getHeight()==frame.getWidth())
				{
					Square square = new Square(frame,true);
					vFigures.remove(i);
					vFigures.add(i, square);
				}
			}
			i++;
		}
	}




	@Override
	public void actionsOnterminatedPsCustom(PSTricksParameters param) throws NotFullyManagedException
	{
		Figure f;
		int i=0;
		
		while(!psCustFigures.isEmpty())
		{
			f = psCustFigures.remove(0);
			
			if(!(f instanceof AkinPoints) || ((AkinPoints)f).getNbPoints()>1)
			{
				if(f instanceof AkinPoints && ((AkinPoints)f).getType()==AkinPoints.TYPE_CURVES)
				{// We must transform the read points.
					AkinPoints ak = (AkinPoints)f;
					int j, size = ak.getNbPoints();
					LaTeXDrawPoint2D prev, curr = ak.getPoint(0);
					
					for(j=1; j<size; j++)
					{
						prev = curr;
						curr = ak.getPoint(j);
						curr.setLocation(2*curr.x-prev.x, 2*curr.y-prev.y);
					}
					
					((AkinPoints)f).updateBorders();
				}
				
				figures.add(f);
				i++;
			}
		}
	}




	@Override
	public void actionsOnNewPath(int line)
	{
		psCustFigures.clear();
	}




	@Override
	public int parseOpenShadowCommand(PSTricksParameters param, int line, String txt) throws InvalidFormatCommandException
	{
		PSTricksParameters pstp = new PSTricksParameters(param);
		int end =  super.parseOpenShadowCommand(pstp, line, txt);
		
		for(Figure f : psCustFigures)
		{
			f.setHasShadow(true);
			f.setShadowAngle(Math.toRadians(pstp.shadowAngle));
			f.setShadowColor(pstp.shadowCol);
			f.setShadowSize(Math.abs(pstp.shadowSize)*Figure.PPC);
		}
		
		if(psCustFigures.size()>0 && psCustFigures.lastElement() instanceof AkinPoints)
		{
			double xunit = param.unit!=PSTricksConstants.DEFAULT_UNIT? param.unit : param.xUnit;
			double yunit = param.unit!=PSTricksConstants.DEFAULT_UNIT? param.unit : param.yUnit;
			
			AkinPoints ak = new AkinPoints(new LaTeXDrawPoint2D(
					param.psCustomP.lastPosition.x*Figure.PPC*xunit, 
					param.psCustomP.lastPosition.y*Figure.PPC*yunit*-1), true);
			ak.setInterval(1);
			ak.setType(AkinPoints.TYPE_LINES);
			psCustFigures.add(ak);
		}
		
		return end;
	}




	@Override
	public int actionsOnCurveTo(int line, Vector<PSTricksParameters> params, String txt)
								throws InvalidFormatCommandException
	{
		Point2D.Double p1 = new Point2D.Double();
		Point2D.Double p2 = new Point2D.Double();
		Point2D.Double p3 = new Point2D.Double();
		int end = parseParametersCurveTo(line, txt, p1, p2, p3);
		PSTricksParameters last = params.lastElement();
		AkinPoints ak ;
		double xunit = last.unit!=PSTricksConstants.DEFAULT_UNIT? last.unit : last.xUnit;
		double yunit = last.unit!=PSTricksConstants.DEFAULT_UNIT? last.unit : last.yUnit;
		
		if(psCustFigures.isEmpty() || !(psCustFigures.lastElement() instanceof AkinPoints))
		{
			ak = new AkinPoints(new LaTeXDrawPoint2D(last.psCustomP.lastPosition.x*Figure.PPC*xunit, 
					last.psCustomP.lastPosition.y*Figure.PPC*-1*yunit), true);
			setFigureParameters(params.lastElement(), ak, false);
			ak.setInterval(1);
			psCustFigures.add(ak);
		}
		else ak = (AkinPoints)psCustFigures.lastElement();
		
		ak.setType(AkinPoints.TYPE_CURVES);
		ak.addPoint(new LaTeXDrawPoint2D(p3.x*Figure.PPC*xunit, p3.y*Figure.PPC*-1*yunit));
		last.psCustomP.lastPosition.setLocation(p3);

		return end;
	}




	@Override
	public int actionsOnLineTo(int line, Vector<PSTricksParameters> params, String txt) throws InvalidFormatCommandException
	{
		Point2D.Double p1 = new Point2D.Double();
		int end = parseParametersLineMoveTo(line, txt, p1);
		PSTricksParameters last = params.lastElement();
		AkinPoints ak ;
		double xunit = last.unit!=PSTricksConstants.DEFAULT_UNIT? last.unit : last.xUnit;
		double yunit = last.unit!=PSTricksConstants.DEFAULT_UNIT? last.unit : last.yUnit;
		
		if(psCustFigures.isEmpty() || !(psCustFigures.lastElement() instanceof AkinPoints))
		{
			ak = new AkinPoints(new LaTeXDrawPoint2D(last.psCustomP.lastPosition.x*Figure.PPC*xunit, 
								last.psCustomP.lastPosition.y*Figure.PPC*-1*yunit), true);
			setFigureParameters(params.lastElement(), ak, false);
			ak.setInterval(1);
			ak.setType(AkinPoints.TYPE_LINES);
			psCustFigures.add(ak);
		}
		else ak = (AkinPoints)psCustFigures.lastElement();
		
		ak.addPoint(new LaTeXDrawPoint2D(p1.x*Figure.PPC*xunit, p1.y*Figure.PPC*-1*yunit));
		last.psCustomP.lastPosition.setLocation(p1);
		
		return end;
	}




	@Override
	public int actionsOnMoveTo(int line, Vector<PSTricksParameters> params, String txt) throws InvalidFormatCommandException
	{
		Point2D.Double p1 = new Point2D.Double();
		int end = parseParametersLineMoveTo(line, txt, p1);
		PSTricksParameters last = params.lastElement();
		last.psCustomP.lastPosition.setLocation(p1);
		
		return end;
	}




	@Override
	public void actionsOnClosePathCommand(int line)
	{
		if(!psCustFigures.isEmpty())
		{
			Figure f = psCustFigures.lastElement();
			
			if(f instanceof AkinPoints)
				((AkinPoints)f).setOpen(false);
		}
	}
	
}


