/***************************************************************************
 *
 *  $Id: kzenexplorerview.cpp,v 1.36 2005/07/03 18:27:52 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include "qlayout.h"
#include "qheader.h"
#include "qregexp.h"
#include "qstring.h"
#include "qlistview.h"

#include <kapplication.h>
#include <kmainwindow.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdebug.h>

#include "kzenconfig.h"
#include "kzenglobal.h"
#include "kzenitem.h"
#include "kzenexplorer.h"
#include "kzenexplorerview.h"

QWidget * fgTrackSource = 0;
QWidget * fgNaviSource = 0;

KZenExplorerView::KZenExplorerView(KZenExplorer * parent, const char *name)
  :QWidget(parent, name)
{
  fExplorer = parent;

  fPlayList = 0;

  fTrackContextMenu = 0;

  fNaviContextMenuJukebox = 0;
  fNaviContextMenuAlbum = 0;
  fNaviContextMenuPlayList = 0;
  fNaviContextMenuSmartPlayList = 0;
  
  QVBoxLayout * l = new QVBoxLayout(this);
  
  QVBox * v = new QVBox(this);

  QHBox * h = new QHBox(v);

  fNaviBar = new KMultiTabBar(KMultiTabBar::Vertical, h, "NaviBar");
  fNaviBar->setStyle(KMultiTabBar::VSNET);
  fNaviBar->setPosition(KMultiTabBar::Left);
  fNaviBar->showActiveTabTexts(true);
  fNaviBar->setFixedWidth(fNaviBar->sizeHint().width()+5);
  fNaviBar->move(0, 3);  

  fNaviBar->appendTab(UserIcon("jukebox", 16), KZen::TabAll, i18n("Jukebox"));
  connect(fNaviBar->tab(KZen::TabAll), SIGNAL(clicked(int)),
	  this, SLOT(naviTabClicked(int)));

  fNaviBar->appendTab(UserIcon("artist", 16), KZen::TabArtists, i18n("Artists"));
  connect(fNaviBar->tab(KZen::TabArtists), SIGNAL(clicked(int)),
	  this, SLOT(naviTabClicked(int)));

  fNaviBar->appendTab(UserIcon("album", 16), KZen::TabAlbums, i18n("Albums"));
  connect(fNaviBar->tab(KZen::TabAlbums), SIGNAL(clicked(int)),
	  this, SLOT(naviTabClicked(int)));

  fNaviBar->appendTab(UserIcon("genre", 16), KZen::TabGenres, i18n("Genres"));
  connect(fNaviBar->tab(KZen::TabGenres), SIGNAL(clicked(int)),
	  this, SLOT(naviTabClicked(int)));

  fNaviBar->appendTab(UserIcon("playlist", 16), KZen::TabPlayLists, i18n("Playlists"));
  connect(fNaviBar->tab(KZen::TabPlayLists), SIGNAL(clicked(int)),
	  this, SLOT(naviTabClicked(int)));

  fMainSplitter = new QSplitter(h);
  fMainSplitter->setOrientation(Qt::Horizontal);
  fMainSplitter->setChildrenCollapsible(false);
  fMainSplitter->setOpaqueResize(true);

  fNaviView = new KZenNavigationView(fMainSplitter, "NaviView");
  fgNaviSource = fNaviView;
  fNaviView->setAllColumnsShowFocus(true);
  fNaviView->addColumn("Navigation");
  fNaviView->setFullWidth(true);
  fNaviView->setRootIsDecorated(false);
  fNaviView->setSorting(0);
  fNaviView->setAcceptDrops(true);
  fNaviView->header()->hide();
  fMainSplitter->setResizeMode(fNaviView, QSplitter::KeepSize);
  
  fTrackView = new KZenTrackView(fExplorer, this, fMainSplitter, "TrackView");

  fUsageBar = new KZenUsageBar(v, "usage");
  fUsageBar->setCenterIndicator(false);
  fUsageBar->setPercentageVisible(true);
  fUsageBar->setFixedHeight(fUsageBar->sizeHint().height()+5);
  fUsageBar->move(0,3);  

  l->addWidget(v);

  fTrackView->setViewMode(ViewModeAll);

  connect(fNaviView, SIGNAL(selectionChanged(QListViewItem *)),
	  this, SLOT(selectionChangedNavi(QListViewItem *)));
  connect(fNaviView, SIGNAL(selectionChanged(QListViewItem *)),
	  fExplorer, SLOT(selectionChangedNavi(QListViewItem *)));
  connect(fNaviView, SIGNAL(filesDropped(const QStringList &, KZenPlayList *)),
	  fExplorer, SLOT(filesDropped(const QStringList &, KZenPlayList *)));
  connect(fNaviView, SIGNAL(tracksDropped(const QStringList &, KZenPlayList *)),
	  fExplorer, SLOT(tracksDropped(const QStringList &, KZenPlayList *)));
  
  connect(fNaviView, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
	  this, SLOT(contextMenuNavi(QListViewItem *, const QPoint &, int)));

  fCurrentNaviItem = 0;

  fCurrentNaviTab = -1;

  fCurrentTrackList = 0;

  fBusy = false;
}

KZenExplorerView::~KZenExplorerView()
{
  fNaviView->clear();
  delete fNaviView;
  fgNaviSource = 0;

  delete fMainSplitter;
}

int KZenExplorerView::getNaviWidth()
{
  if (fNaviView->isShown()) {
    QValueList<int> sizes = fMainSplitter->sizes();
    return sizes.first();
  } else {
    return fNaviWidth; 
  }
}

void KZenExplorerView::setNaviWidth(int w)
{
  QValueList<int> sizes;
  sizes.append(w);
  sizes.append(w);
  fMainSplitter->setSizes(sizes);
}

void KZenExplorerView::setupActions()
{
  KActionCollection * actionCollection =
    static_cast<KMainWindow *>(kapp->mainWidget())->actionCollection();
  
  fPlayTracks = actionCollection->action("play_tracks");
  fQueueTracks = actionCollection->action("queue_tracks");
  fStopPlay = actionCollection->action("stop_play");
  fEditTracks = actionCollection->action("edit_tracks");
  fSaveTracks = actionCollection->action("save_tracks");
  fDeleteTracks = actionCollection->action("delete_tracks");
  fDestroyTracks = actionCollection->action("destroy_tracks");
  fIsCompilation = actionCollection->action("iscompilation");
  fShowDetails = actionCollection->action("showdetails");

  fNewPlayList = actionCollection->action("new_playlist");
  fNewSmartPlayList = actionCollection->action("new_smart_playlist");
  fRenamePlayList = actionCollection->action("rename_playlist");
  fEditSmartPlayList = actionCollection->action("edit_smart_playlist");
  fShufflePlayList = actionCollection->action("shuffle_playlist");
  fUpdatePlayList = actionCollection->action("update_playlist");
  fDeletePlayList = actionCollection->action("delete_playlist");
  fDestroy = actionCollection->action("destroy");
  fSelectAll = actionCollection->action("select_all");
  fPlayListFromSelection = actionCollection->action("new_playlist_selection");
  
  fTrackContextMenu = new KPopupMenu(this);
  fPlayTracks->plug(fTrackContextMenu);
  fQueueTracks->plug(fTrackContextMenu);
  fStopPlay->plug(fTrackContextMenu);
  fTrackContextMenu->insertSeparator();
  fSaveTracks->plug(fTrackContextMenu);
  fTrackContextMenu->insertSeparator();
  fEditTracks->plug(fTrackContextMenu);
  fSelectAll->plug(fTrackContextMenu);
  fPlayListFromSelection->plug(fTrackContextMenu);
  fTrackContextMenu->insertSeparator();
  fDeleteTracks->plug(fTrackContextMenu);
  fDestroyTracks->plug(fTrackContextMenu);

  fNaviContextMenuJukebox = new KPopupMenu(this);
  fNewPlayList->plug(fNaviContextMenuJukebox);
  fNewSmartPlayList->plug(fNaviContextMenuJukebox);
  fNaviContextMenuJukebox->insertSeparator();
  fDestroy->plug(fNaviContextMenuJukebox);
  fNaviContextMenuJukebox->insertSeparator();
  fShowDetails->plug(fNaviContextMenuJukebox);

  fNaviContextMenuAlbum = new KPopupMenu(this);
  fNewPlayList->plug(fNaviContextMenuAlbum);
  fNewSmartPlayList->plug(fNaviContextMenuAlbum);
  fNaviContextMenuAlbum->insertSeparator();
  fDestroy->plug(fNaviContextMenuAlbum);
  fNaviContextMenuAlbum->insertSeparator();
  fIsCompilation->plug(fNaviContextMenuAlbum);
  fNaviContextMenuAlbum->insertSeparator();
  fShowDetails->plug(fNaviContextMenuAlbum);

  fNaviContextMenuPlayList = new KPopupMenu(this);
  fNewPlayList->plug(fNaviContextMenuPlayList);
  fNewSmartPlayList->plug(fNaviContextMenuPlayList);
  fNaviContextMenuPlayList->insertSeparator();
  fRenamePlayList->plug(fNaviContextMenuPlayList);
  fShufflePlayList->plug(fNaviContextMenuPlayList);
  fUpdatePlayList->plug(fNaviContextMenuPlayList);
  fNaviContextMenuPlayList->insertSeparator();
  fDeletePlayList->plug(fNaviContextMenuPlayList);
  fDestroy->plug(fNaviContextMenuPlayList);
  fNaviContextMenuPlayList->insertSeparator();
  fShowDetails->plug(fNaviContextMenuPlayList);
  
  fNaviContextMenuSmartPlayList = new KPopupMenu(this);
  fNewPlayList->plug(fNaviContextMenuSmartPlayList);
  fNewSmartPlayList->plug(fNaviContextMenuSmartPlayList);
  fNaviContextMenuSmartPlayList->insertSeparator();
  fRenamePlayList->plug(fNaviContextMenuSmartPlayList);
  fEditSmartPlayList->plug(fNaviContextMenuSmartPlayList);
  fShufflePlayList->plug(fNaviContextMenuSmartPlayList);
  fUpdatePlayList->plug(fNaviContextMenuSmartPlayList);
  fNaviContextMenuSmartPlayList->insertSeparator();
  fDeletePlayList->plug(fNaviContextMenuSmartPlayList);
  fDestroy->plug(fNaviContextMenuSmartPlayList);
  fNaviContextMenuSmartPlayList->insertSeparator();
  fShowDetails->plug(fNaviContextMenuSmartPlayList);

  fTrackView->setupActions();
}

void KZenExplorerView::displayTrackList(QPtrList<KZenTrack> * list, ViewMode vm)
{
  fPlayList = 0;
  fViewMode = vm;

  if (vm==ViewModeSearch) {
    fNaviView->selectAll(false);
  }

  fCurrentNaviItem = (KZenItem*) fNaviView->selectedItem();
  fTrackView->displayTrackList(list, vm, fCurrentNaviItem);

  fCurrentTrackList = list;
}

void KZenExplorerView::displayTrackListPL(KZenPlayList * PL)
{
  fPlayList = PL;
  fViewMode = ViewModePlayList;
  
  fTrackView->displayTrackListPL(PL);

  fCurrentTrackList = PL->getTrackList();
}

void KZenExplorerView::displayArtistList(QPtrList<KZenArtist> * list)
{
  KZenTrack * track;
  KZenArtist * artist;
  KZenAlbum * album;
  KZenArtistItem * artistitem;
  KZenAlbumItem * albumitem;
  for (uint i=0;i<list->count();i++) {
    artist = list->at(i);
    artistitem = new KZenArtistItem(fNaviView, artist);
    artist->setItem(artistitem);

    for (uint a=0;a<artist->getAlbumList()->count();a++) {
      album = artist->getAlbumList()->at(a);
      albumitem = new KZenAlbumItem(artistitem, album);
      album->setItem(albumitem);
      
      for (uint t=0;t<album->getTrackList()->count();t++) {
	track = album->getTrackList()->at(t);
	albumitem->setIsCompilation(track->getIsCompilation());
      }
    }

    fArtistItems.append(artistitem);

    for (uint t=0;t<artist->getTrackList()->count();t++) {
      track = artist->getTrackList()->at(t);
     
      KZenAlbumItemTL * albumitemTL = 0;
      for (uint al=0;al<fAlbumItems.count();al++) {
	albumitemTL = fAlbumItems.at(al);
	if (albumitemTL->text(0)==track->getAlbum() &&
	    (albumitemTL->text(1)==track->getArtist() ||
	     albumitemTL->getIsCompilation())) break;
	albumitemTL = 0;
      }
      if (!albumitemTL) {
	albumitemTL = new KZenAlbumItemTL(fNaviView, track->getAlbum());
	albumitemTL->setText(1,track->getArtist());
	fAlbumItems.append(albumitemTL);
      }
      if (track->getIsCompilation()) albumitemTL->setIsCompilation(true);
      albumitemTL->addTrack(track);
    }
  }
}

void KZenExplorerView::displayGenreList(QPtrList<KZenGenre> * list)
{
  KZenGenre * genre;
  KZenGenreItem * genreitem;
  for (uint i=0;i<list->count();i++) {
    genre = list->at(i);
    genreitem = new KZenGenreItem(fNaviView, genre);
    genre->setItem(genreitem);
    fGenreItems.append(genreitem);
  }
}

void KZenExplorerView::displayPlayLists(QPtrList<KZenPlayList> * list,
					QPtrList<KZenSmartPlayList> * slist)
{
  KZenPlayList * playlist;
  KZenPlayListItem * playlistitem;
  for (uint i=0;i<list->count();i++) {
    playlist = list->at(i);
    playlistitem = new KZenPlayListItem(fNaviView, playlist);
    fPlayListItems.append(playlistitem);
  }
  KZenSmartPlayList * splaylist;
  KZenSmartPlayListItem * splaylistitem;
  for (uint i=0;i<slist->count();i++) {
    splaylist = slist->at(i);
    splaylistitem = new KZenSmartPlayListItem(fNaviView, splaylist);
    fPlayListItems.append(splaylistitem);
  }  
}

void KZenExplorerView::trackUpdated(KZenTrack * track)
{
  kdDebug() << "trackUpdated" << endl;

  KZenItem * item = (KZenItem*) fNaviView->selectedItem();

  KZenArtist * artist = track->getZenArtist();
  KZenAlbum * album = track->getZenAlbum();
  artist->getTrackList()->remove(track);
  if (artist->getItem() && artist->getItem()==item) fNaviView->selectAll(false);
  if (album->getItem() && album->getItem()==item) fNaviView->selectAll(false);
  if (artist->getTrackList()->count()==0) {
    fArtistItems.remove(artist->getItem());
    delete artist->getItem();
    fExplorer->getArtists()->remove(artist);
    artist = 0;
    kdDebug() << "delete artist" << endl;
  } else {
    album->getTrackList()->remove(track);
    if (album->getTrackList()->count()==0) {
      delete album->getItem();
      artist->getAlbumList()->remove(album);
      kdDebug() << "delete album" << endl;
    }
  }

  artist = 0;
  if (track->getIsCompilation()) {
    artist = fExplorer->getVariousArtists();
  } else {
    for (uint i=0;i<fExplorer->getArtists()->count();i++) {
      artist = fExplorer->getArtists()->at(i);
      if (artist->getName()==track->getArtist()) break;
      artist = 0;
    }

    if (!artist) {
      kdDebug() << "new artist" << endl;
      artist = new KZenArtist(track->getArtist());
      fExplorer->getArtists()->append(artist);
      KZenArtistItem * artistitem = new KZenArtistItem(fNaviView, artist);
      artist->setItem(artistitem);
      fArtistItems.append(artistitem);
      if (fCurrentNaviTab!=KZen::TabArtists) artistitem->setVisible(false);
    }
  }
  track->setZenArtist(artist);
  artist->addTrack(track);
  
  album = track->getZenAlbum();
  if (album->getItem()==0) {
    kdDebug() << "new album item" << endl;
    KZenAlbumItem * albumitem = new KZenAlbumItem(artist->getItem(), album);
    album->setItem(albumitem);
  }

  KZenAlbumItemTL * albumitemTL = 0;
  for (uint i=0;i<fAlbumItems.count();i++) {
    albumitemTL = fAlbumItems.at(i);
    if (albumitemTL->getTrackList()->find(track)!=-1) break;
    albumitemTL = 0;
  }

  if (albumitemTL) {
    albumitemTL->getTrackList()->remove(track);
    if (albumitemTL->getTrackList()->count()==0) {
      if (albumitemTL==item) fNaviView->selectAll(false);
      fAlbumItems.remove(albumitemTL);
      delete albumitemTL;
    }
  }

  albumitemTL = 0;
  for (uint i=0;i<fAlbumItems.count();i++) {
    albumitemTL = fAlbumItems.at(i);
    if (albumitemTL->text(0)==track->getAlbum() &&
	(albumitemTL->text(1)==track->getArtist() ||
	 track->getIsCompilation())) break;
    albumitemTL = 0;
  }

  if (!albumitemTL) {
    albumitemTL = new KZenAlbumItemTL(fNaviView,track->getAlbum());
    albumitemTL->setText(1,track->getArtist());
    fAlbumItems.append(albumitemTL);
    if (fCurrentNaviTab!=KZen::TabAlbums) albumitemTL->setVisible(false);
  }
  if (track->getIsCompilation()) albumitemTL->setIsCompilation(true);
  albumitemTL->addTrack(track);

  KZenGenre * genre = track->getZenGenre();
  genre->getTrackList()->remove(track);
  if (genre->getTrackList()->count()==0) {
    if (genre->getItem() && genre->getItem()==item) fNaviView->selectAll(false);
    fGenreItems.remove(genre->getItem());
    delete genre->getItem();
    fExplorer->getGenres()->remove(genre);
  }
  
  genre = 0;
  for (uint i=0;i<fExplorer->getGenres()->count();i++) {
    genre = fExplorer->getGenres()->at(i);
    if (genre->getName()==track->getGenre()) break;
    genre = 0;
  }

  if (!genre) {
    genre = new KZenGenre(track->getGenre());
    fExplorer->getGenres()->append(genre);
    KZenGenreItem * genreitem = new KZenGenreItem(fNaviView,genre);
    fGenreItems.append(genreitem);
    if (fCurrentNaviTab!=KZen::TabGenres) genreitem->setVisible(false);
    genre->setItem(genreitem);
  }
  genre->addTrack(track);
  track->setZenGenre(genre);

  fNaviView->triggerUpdate();
}

void KZenExplorerView::addTrack(KZenTrack * track)
{  
  KZenArtistItem * artistitem = 0;
  for (uint i=0;i<fArtistItems.count();i++) {
    artistitem = fArtistItems.at(i);
    if (artistitem->text(0)==track->getArtist()) break;
    artistitem = 0;
  }
  if (!artistitem) {
    artistitem = new KZenArtistItem(fNaviView, track->getZenArtist());
    track->getZenArtist()->setItem(artistitem);
    fArtistItems.append(artistitem);
    if (fCurrentNaviTab!=KZen::TabArtists) artistitem->setVisible(false);
  }

  KZenAlbumItem * albumitem = (KZenAlbumItem*)artistitem->firstChild();
  while (albumitem) {
    if (albumitem->text(0)==track->getAlbum()) break;
    albumitem = (KZenAlbumItem *)albumitem->nextSibling();
  }
  if (!albumitem || albumitem->text(0)!=track->getAlbum()) {
    kdDebug() << track->getZenAlbum() << endl;
    albumitem = new KZenAlbumItem(artistitem, track->getZenAlbum());
    track->getZenAlbum()->setItem(albumitem);
  }

  KZenAlbumItemTL * albumitemTL = 0;
  for (uint i=0;i<fAlbumItems.count();i++) {
    albumitemTL = fAlbumItems.at(i);
    if (albumitemTL->text(0)==track->getAlbum() &&
	(albumitemTL->text(1)==track->getArtist() ||
	 albumitemTL->getIsCompilation())) break;
    albumitemTL = 0;
  }
  if (!albumitemTL) {
    albumitemTL = new KZenAlbumItemTL(fNaviView, track->getAlbum());
    albumitemTL->setText(1,track->getArtist());
    fAlbumItems.append(albumitemTL);
    if (fCurrentNaviTab!=KZen::TabAlbums) albumitemTL->setVisible(false);
  }
  if (track->getIsCompilation()) albumitemTL->setIsCompilation(true);
  albumitemTL->addTrack(track);
  
  KZenGenreItem * genreitem = 0;
  for (uint i=0;i<fGenreItems.count();i++) {
    genreitem = fGenreItems.at(i);
    if (genreitem->text(0)==track->getGenre()) break;
    genreitem = 0;
  }
  if (!genreitem) {
    genreitem = new KZenGenreItem(fNaviView, track->getZenGenre());
    fGenreItems.append(genreitem);
    if (fCurrentNaviTab!=KZen::TabGenres) genreitem->setVisible(false);
  }
  track->getZenGenre()->setItem(genreitem);

  if (fViewMode==ViewModeAll) {
    fTrackView->addTrack(track);
  }
  if (fViewMode==ViewModeArtist) {
    KZenArtistItem * artistitem = ((KZenArtistItem *)fNaviView->selectedItem());
    if (artistitem) {
      KZenArtist * artist = artistitem->getArtist();
      if (artist->getName()==track->getArtist()) {
	fTrackView->addTrack(track);
      }
    }
  }
  if (fViewMode==ViewModeAlbum) {
    KZenAlbumItem * albumitem = ((KZenAlbumItem *)fNaviView->selectedItem());
    if (albumitem) {
      KZenAlbum * album = albumitem->getAlbum();
      if (album->getArtist()->getName()==track->getArtist() &&
	  album->getName()==track->getAlbum()) {
	fTrackView->addTrack(track);
      }
    }
  }
  if (fViewMode==ViewModePlayList) {
    fTrackView->addTrack(track, fPlayList);
  }
}

void KZenExplorerView::removeTrackItem(KZenTrackItem * trackitem)
{
  delete trackitem;
}

void KZenExplorerView::contextMenuTracks(QListViewItem * item, const QPoint & pos, int)
{
  if (!item) return;
  
  if (!fBusy) {
    fPlayTracks->setEnabled(true);
    fQueueTracks->setEnabled(true);
    fEditTracks->setEnabled(true);
    if (fViewMode==ViewModePlayList) {
      fDeleteTracks->setEnabled(true);
    } else {
      fDeleteTracks->setEnabled(false);
    }
    fDestroyTracks->setEnabled(true);
  } else {
    fPlayTracks->setEnabled(false);
    fQueueTracks->setEnabled(false);
    fEditTracks->setEnabled(false);
    fDeleteTracks->setEnabled(false);
    fDestroyTracks->setEnabled(false);
  }
  
  fTrackContextMenu->popup(pos);
}

void KZenExplorerView::contextMenuNavi(QListViewItem * item, const QPoint & pos, int)
{
  if (fBusy) return;

  KZenItem * zenI = (KZenItem*)item;
  if (zenI) {
    if (zenI->getItemType()==KZenItem::ItemTypePlayList) {

      fDestroy->setEnabled(true);
      
      if (KZenConfig::get()->showDetailsPlayList()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);
      
      fNaviContextMenuPlayList->popup(pos);

    } else if (zenI->getItemType()==KZenItem::ItemTypeSmartPlayList) {

      fDestroy->setEnabled(true);

      if (KZenConfig::get()->showDetailsPlayList()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);

      fNaviContextMenuSmartPlayList->popup(pos);

    } else if (zenI->getItemType()==KZenItem::ItemTypeAlbum) {
    
      fDestroy->setEnabled(true);
      
      fIsCompilation->setText(i18n("Mark as Compilation"));
      fIsCompilation->setIcon("compilation.png");
      fIsCompilation->setEnabled(false);

      if (KZenConfig::get()->showDetailsAlbum()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);
      
      fNaviContextMenuAlbum->popup(pos);

    } else if (zenI->getItemType()==KZenItem::ItemTypeAlbumTL) {
    
      fDestroy->setEnabled(true);
      
      KZenAlbumItemTL * albumitem = (KZenAlbumItemTL*)zenI;
      if (albumitem->getIsCompilation()) {
	fIsCompilation->setText(i18n("Unmark as Compilation"));
	fIsCompilation->setIcon("uncompilation.png");
      } else {
	fIsCompilation->setText(i18n("Mark as Compilation"));
	fIsCompilation->setIcon("compilation.png");
      }
      fIsCompilation->setEnabled(true);

      if (KZenConfig::get()->showDetailsAlbum()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);
      
      fNaviContextMenuAlbum->popup(pos);

    } else if (zenI->getItemType()==KZenItem::ItemTypeArtist) {
      
      fDestroy->setEnabled(true);

      if (KZenConfig::get()->showDetailsArtist()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);
      
      fNaviContextMenuJukebox->popup(pos);
    
    } else if (zenI->getItemType()==KZenItem::ItemTypeGenre) {
      
      fDestroy->setEnabled(true);

      if (KZenConfig::get()->showDetailsGenre()) {
	fShowDetails->setText(i18n("Hide Details"));
      } else {
	fShowDetails->setText(i18n("Show Details"));	
      }
      fShowDetails->setEnabled(true);
      
      fNaviContextMenuJukebox->popup(pos);
    
    } else {
      
      fDestroy->setEnabled(false);
      fShowDetails->setEnabled(false);
      
      fNaviContextMenuJukebox->popup(pos);
      
    }
  } else {
    
    fDestroy->setEnabled(false);
    fShowDetails->setEnabled(false);
    
    fNaviContextMenuJukebox->popup(pos);
  }
}

QPtrList<KZenTrackItem> & KZenExplorerView::selectedTracks()
{
  return fTrackView->selectedTracks();
}

void KZenExplorerView::refresh()
{
  fNaviView->triggerUpdate();
  fTrackView->refresh();
}

void KZenExplorerView::playTracks()
{
  fExplorer->playTracks(fTrackView->selectedTracks());
}

void KZenExplorerView::queueTracks()
{
  fExplorer->queueTracks(fTrackView->selectedTracks());
}

void KZenExplorerView::editTracks()
{
  fExplorer->editTracks(fTrackView->selectedTracks());
}

void KZenExplorerView::saveTracks()
{
  fExplorer->saveTracks(fTrackView->selectedTracks());
}

void KZenExplorerView::deleteTracks()
{
  emit deleteTracks(fTrackView->selectedTracks(), fPlayList);
}

void KZenExplorerView::destroyTracks()
{
  emit destroyTracks(fTrackView->selectedTracks());
}

void KZenExplorerView::renamePlayList()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && 
      (item->getItemType()==KZenItem::ItemTypePlayList ||
       item->getItemType()==KZenItem::ItemTypeSmartPlayList)) {
    fExplorer->renamePlayList((KZenPlayListItem*)item);
  }
}

void KZenExplorerView::editSmartPlayList()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && item->getItemType()==KZenItem::ItemTypeSmartPlayList) {
    fExplorer->editSmartPlayList((KZenSmartPlayListItem*)item);
  }
}

void KZenExplorerView::shufflePlayList()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && 
      (item->getItemType()==KZenItem::ItemTypePlayList ||
       item->getItemType()==KZenItem::ItemTypeSmartPlayList)) {
    KZenPlayList * pl = ((KZenPlayListItem*)item)->getPlayList();
    pl->shuffle();
    fUpdatePlayList->setEnabled(true);
    displayTrackListPL(pl);
  }
}

void KZenExplorerView::updatePlayList()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item &&
      (item->getItemType()==KZenItem::ItemTypePlayList ||
       item->getItemType()==KZenItem::ItemTypeSmartPlayList)) {
    KZenPlayList * pl = ((KZenPlayListItem*)item)->getPlayList();
    if (pl->isModified()) {
      fExplorer->updatePlayList(((KZenPlayListItem*)item)->getPlayList());
    }
  }
}

void KZenExplorerView::deleteNaviItem()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && item->getItemType()!=KZenItem::ItemTypeUnknown) {
    fExplorer->deleteNaviItem(item);
  }
  fCurrentNaviItem = 0;
}

void KZenExplorerView::destroyNaviItem()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && item->getItemType()!=KZenItem::ItemTypeUnknown) {
    fExplorer->destroyNaviItem(item);
  }
  fCurrentNaviItem = 0;
}

void KZenExplorerView::addPlayList(KZenPlayList * pl, bool showtab)
{ 
  KZenPlayListItem * item = new KZenPlayListItem(fNaviView, pl);
  fPlayListItems.append(item);
  if (showtab) fNaviBar->setTab(KZen::TabPlayLists, true);
}

void KZenExplorerView::addPlayList(KZenSmartPlayList * pl, bool showtab)
{
  KZenSmartPlayListItem * item = new KZenSmartPlayListItem(fNaviView, pl);
  fPlayListItems.append(item);
  if (showtab) fNaviBar->setTab(KZen::TabPlayLists, true);
}

void KZenExplorerView::removeNaviItem(KZenItem * item)
{  
  fArtistItems.remove((KZenArtistItem*)item);
  fPlayListItems.remove((KZenPlayListItem*)item);
  if (fCurrentNaviItem==item) fCurrentNaviItem = 0;
  delete item;
}

void KZenExplorerView::selectNaviItem(KZenItem * item)
{  
  if (item)
    fNaviView->setSelected(item, true);

  fCurrentNaviItem = item;
}

void KZenExplorerView::removeArtist(const char * name)
{
  kdDebug() << "removeArtist" << endl;

  KZenItem * item = (KZenItem *) fNaviView->firstChild();
  while (item) {
    if (item->getItemType()==KZenItem::ItemTypeArtist &&
	item->text(0)==name) {
      fArtistItems.remove((KZenArtistItem*)item);
      delete item;
      break;
    }
    item  = (KZenItem*) item->nextSibling();
  }
}

void KZenExplorerView::removeGenre(const char * name)
{
  kdDebug() << "removeGenre" << endl;

  KZenItem * item = (KZenItem *) fNaviView->firstChild();
  while (item) {
    if (item->getItemType()==KZenItem::ItemTypeGenre &&
	item->text(0)==name) {
      fGenreItems.remove((KZenGenreItem*)item);
      delete item;
      break;
    }
    item  = (KZenItem*) item->nextSibling();
  }
}

void KZenExplorerView::removeAlbum(const char * arname,const char * alname)
{
  kdDebug() << "removeAlbum" << endl;

  KZenItem * item = (KZenItem *) fNaviView->firstChild();
  while (item) {
    if (item->getItemType()==KZenItem::ItemTypeArtist &&
	item->text(0)==arname) {
      KZenAlbumItem * album = (KZenAlbumItem*) item->firstChild();
      while (album) {
	if (album->text(0)==alname) {
	  delete album;
	  break;
	}
	album = (KZenAlbumItem*) album->nextSibling();
      }
    }

    if (item->getItemType()==KZenItem::ItemTypeAlbumTL) {
      KZenAlbumItemTL * album = (KZenAlbumItemTL*) item;
      if (album->text(0)==alname &&
	  (album->text(1)==arname ||
	   album->getIsCompilation())) {
	fAlbumItems.remove(album);
	delete album;
      }
    }
    
    item  = (KZenItem*) item->nextSibling();
  }
}

void KZenExplorerView::setSearchText(const char * text)
{
  fTrackView->setSearchText(text);
}

void KZenExplorerView::setSortColumn(KZen::TagField c)
{
  fTrackView->setSortColumn(c);
}

void KZenExplorerView::setUsage(int used, int total)
{
  fUsageBar->setProgress(used, total);
}

void KZenExplorerView::setBusy(bool busy)
{
  fBusy = busy;

  fNaviBar->setEnabled(!fBusy);
  fNaviView->setEnabled(!fBusy);
  fTrackView->setBusy(fBusy);
}

void KZenExplorerView::selectAll()
{
  fTrackView->selectAll();
}

void KZenExplorerView::naviTabClicked(int i)
{
  if (fBusy) return;

  if (fCurrentNaviTab==i) {
    fNaviBar->setTab(i, true);
  } else {
    if (fCurrentNaviTab!=-1) fNaviBar->setTab(fCurrentNaviTab, false);
    fNaviBar->setTab(i, true);

    if (i==KZen::TabAll) {
      fNaviWidth = fMainSplitter->sizes().first();
      fNaviView->hide();
      fNaviView->selectAll(false);

      setSearchText("");
      
      displayTrackList(fExplorer->getTracks(), ViewModeAll);
      fExplorer->setDestroyEnabled(false);
      fExplorer->setUpdateEnabled(false);
      fExplorer->setTrackOpsEnabled(false);
      fExplorer->setPlayListOpsEnabled(false);
      fExplorer->setSmartPlayListOpsEnabled(false);
    } else {
      fNaviView->show();
      fNaviView->selectAll(false);

      if (i==KZen::TabArtists) {
	setArtistsVisible(true);
	setAlbumsVisible(false);
	setGenresVisible(false);
	setPlayListsVisible(false);
      }

      if (i==KZen::TabAlbums) {
	setArtistsVisible(false);
	setAlbumsVisible(true);
	setGenresVisible(false);
	setPlayListsVisible(false);
      }

      if (i==KZen::TabGenres) {
	setArtistsVisible(false);
	setAlbumsVisible(false);
	setGenresVisible(true);
	setPlayListsVisible(false);
      }

      if (i==KZen::TabPlayLists) {
	setArtistsVisible(false);
	setAlbumsVisible(false);
	setGenresVisible(false);
	setPlayListsVisible(true);
      }

      if (fCurrentNaviItem) {
	kdDebug() << "ensureItemVisible" << endl;
	fNaviView->ensureItemVisible(fCurrentNaviItem);
      }

    }
  }

  fCurrentNaviTab = i;
}

void KZenExplorerView::setArtistsVisible(bool visible)
{
  KZenArtistItem * artistitem = fArtistItems.first();
  while (artistitem) {
    artistitem->setVisible(visible);
    artistitem = fArtistItems.next();
  }
}

void KZenExplorerView::setAlbumsVisible(bool visible)
{
  KZenAlbumItemTL * albumitem = fAlbumItems.first();
  while (albumitem) {
    albumitem->setVisible(visible);
    albumitem = fAlbumItems.next();
  }
}

void KZenExplorerView::setGenresVisible(bool visible)
{
  KZenGenreItem * genreitem = fGenreItems.first();
  while (genreitem) {
    genreitem->setVisible(visible);
    genreitem = fGenreItems.next();
  }
}

void KZenExplorerView::setPlayListsVisible(bool visible)
{
  KZenPlayListItem * plitem = fPlayListItems.first();
  while (plitem) {
    plitem->setVisible(visible);
    plitem = fPlayListItems.next();
  }
}

void KZenExplorerView::isCompilationClicked()
{
  KZenItem * item = (KZenItem*) fNaviView->selectedItem();
  if (item && item->getItemType()==KZenItem::ItemTypeAlbum) {
    KZenAlbumItem * albumitem = (KZenAlbumItem *)item;
    kdDebug() << "isCompilationClicked -> " 
	      << albumitem->text(0) << "\t" 
	      << albumitem->getIsCompilation() << endl;
    
    bool comp = albumitem->getIsCompilation();
    albumitem->setIsCompilation(!comp);
  }
}


void KZenExplorerView::showDetails()
{
  KZenItem * zenI = (KZenItem*) fNaviView->selectedItem();
  if (zenI->getItemType()==KZenItem::ItemTypePlayList ||
      zenI->getItemType()==KZenItem::ItemTypeSmartPlayList) {

    bool sd = KZenConfig::get()->showDetailsPlayList();
    KZenConfig::get()->setShowDetailsPlayList(!sd);
    
    fNaviView->triggerUpdate();

  } else if (zenI->getItemType()==KZenItem::ItemTypeAlbum) {

    bool sd = KZenConfig::get()->showDetailsAlbum();
    KZenConfig::get()->setShowDetailsAlbum(!sd);
    
    fNaviView->triggerUpdate();

  } else if (zenI->getItemType()==KZenItem::ItemTypeArtist) {

    bool sd = KZenConfig::get()->showDetailsArtist();
    KZenConfig::get()->setShowDetailsArtist(!sd);
    
    fNaviView->triggerUpdate();
    
  } else if (zenI->getItemType()==KZenItem::ItemTypeGenre) {

    bool sd = KZenConfig::get()->showDetailsGenre();
    KZenConfig::get()->setShowDetailsGenre(!sd);
    
    fNaviView->triggerUpdate();
    
  } else {
    
  }
  
}

void KZenExplorerView::selectionChangedNavi(QListViewItem * i)
{
  fCurrentNaviItem = (KZenItem *) i;
}

#include "kzenexplorerview.moc"
