//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <cstring>
#include <sys/types.h>

#include "silcfiletransfer.h"
#include "silcaccount.h"
#include "silcbuddycontact.h"
#include "silcbuddycliententry.h"

#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <kopeteuiglobal.h>
#include <qfile.h>
#include <kfiledialog.h> 



SilcFileTransfer::SilcFileTransfer(SilcAccount *account,
				   SilcBuddyContact *contact, 
				   const QString &file)
{
  kDebug() << "New outgoing filetransfer" << endl;
  mAccount = account;
  mLocalFile.setFileName(file);
  mContact = contact;
  mFileSize = mLocalFile.size();
  mFileOffset = -1;
  
  mKopeteTransfer = Kopete::TransferManager::transferManager()->addTransfer
    (contact, mLocalFile.fileName(), mLocalFile.size(), contact->contactId(),
     Kopete::FileTransferInfo::Outgoing);

  SilcTK::SilcClientConnectionParams params;
  memset(&params, 0, sizeof(params));

  if (! account->ftNoBind() ) 
    params.local_ip = account->localIp();

  SilcTK::SilcClientFileError err =
    SilcTK::silc_client_file_send(account->client(), account->conn(),
				  mContact->clientEntries()(), &params, 
                                  account->pubkey(), account->privkey(), 
                                  sendFileMonitor, this,
				  file.toLatin1().constData(), &mSid);
  if (err) {
    error(err);
    close();
    deleteLater();
    return;
  }

  connect(mKopeteTransfer, SIGNAL(result(KIO::Job *)),
	  this, SLOT(slotTransferResult()));
}

SilcFileTransfer::SilcFileTransfer(SilcAccount *account,
				   SilcBuddyContact *contact, 
				   SilcTK::SilcUInt32 sid, bool ask_name)
{
  mAccount = account;
  mContact = contact;
  mSid = sid;
  mFileOffset = 0;
  mKopeteTransfer = NULL;
  
  kDebug() << "New incoming filetransfer from "
	   << contact->nickName() << endl;
  
  SilcTK::SilcClientConnectionParams params;
  memset(&params, 0, sizeof(params));
  params.local_ip = account->localIp();
  params.no_authentication = TRUE;

  SilcTK::SilcClientFileError err;
  if (!ask_name)
    err = SilcTK::silc_client_file_receive(account->client(), account->conn(),
                                           &params, account->pubkey(), 
                                           account->privkey(),
                                           sendFileMonitor, this, 
                                           NULL, sid, 0, 0);
  else
    err = SilcTK::silc_client_file_receive
      (account->client(), account->conn(), &params, account->pubkey(), 
       account->privkey(), sendFileMonitor, this, 0, sid, ask_filename, 
       (void *) contact);

}

SilcFileTransfer::~SilcFileTransfer() 
{
}

void 
SilcFileTransfer::error(SilcTK::SilcClientFileError error)
{
  KIO::Error kio_error;
  QString msg;

  switch(error) {
  case SilcTK::SILC_CLIENT_FILE_OK:
    return;

  case SilcTK::SILC_CLIENT_FILE_ERROR:
    msg = i18n("Unable to perform file transfer.");
    kio_error = KIO::ERR_UNKNOWN;
    break;

  case SilcTK::SILC_CLIENT_FILE_UNKNOWN_SESSION:
    msg = i18n("Unknown file transfer session. Unable to perform.");
    kio_error = KIO::ERR_INTERNAL;
    break;

  case SilcTK::SILC_CLIENT_FILE_ALREADY_STARTED:
    msg = i18n("File transfer already started. Stopping here.");
    kio_error = KIO::ERR_INTERNAL;
    break;

  case SilcTK::SILC_CLIENT_FILE_NO_SUCH_FILE:
    msg = i18n("Unable to perform file transfer: no such file.");
    kio_error = KIO::ERR_DOES_NOT_EXIST;
    break;
  
  case SilcTK::SILC_CLIENT_FILE_PERMISSION_DENIED:
    msg = i18n("Unable to perform file transfer: permission denied.");
    kio_error =  KIO::ERR_ACCESS_DENIED;
    break;

  case SilcTK::SILC_CLIENT_FILE_KEY_AGREEMENT_FAILED:
    msg = i18n("Unable to perform file transfer: key agreement failed.");
    kio_error = KIO::ERR_INTERNAL;
    break;

  case SilcTK::SILC_CLIENT_FILE_CONNECT_FAILED:
    msg = i18n("Unable to perform file transfer: error during connect.");
    kio_error = KIO::ERR_COULD_NOT_CONNECT;
    break;

  case SilcTK::SILC_CLIENT_FILE_TIMEOUT:
    msg = i18n("Unable to perform file transfer: connection timeout.");
    kio_error = KIO::ERR_SERVER_TIMEOUT;
    break;

  case SilcTK::SILC_CLIENT_FILE_NO_MEMORY:
    msg = i18n("Unable to performfile transfer: system out of memory.");
    kio_error = KIO::ERR_OUT_OF_MEMORY;
    break;
  }

  mKopeteTransfer->slotError(kio_error, msg);
}

void 
SilcFileTransfer::setOffset(qlonglong offset) 
{
  mFileOffset = offset;
  if(mFileSize == offset) {
    kDebug() << "File Transfer completed" << endl;
    if(mKopeteTransfer)
      mKopeteTransfer->slotComplete();
  } else {
    if (mKopeteTransfer)
      mKopeteTransfer->slotProcessed ( offset );
  }
}

void 
SilcFileTransfer::initTransfer(QString filename, qlonglong filesize) 
{
  mFileSize = filesize;

  mKopeteTransfer = Kopete::TransferManager::transferManager()->addTransfer
    (mContact, filename, (unsigned long)filesize, mContact->contactId(),
    Kopete::FileTransferInfo::Incoming);

  connect(mKopeteTransfer, SIGNAL(result(KIO::Job *)),
	  this, SLOT(slotTransferResult()));
}


bool
SilcFileTransfer::closed() 
{
  if (!mSid)
    return TRUE;

  return FALSE;
}

void 
SilcFileTransfer::close()
{
  if(mSid != 0) {
    SilcTK::silc_client_file_close(mAccount->client(), mAccount->conn(), mSid);
    mSid = 0;
  }
  kDebug() << "FileTransfer closed" << endl;
}

void
SilcFileTransfer::ask_filename(SilcTK::SilcClient client,
                               SilcTK::SilcClientConnection conn,
                               SilcTK::SilcUInt32 session_id,
                               const char *remote_filename,
                               SilcTK::SilcClientFileName completion,
                               void *completion_context,
                               void *context)
{
  SilcBuddyContact *buddy = (SilcBuddyContact *) context;
  QString fileName;
  while(1) {
    fileName =
      KFileDialog::getSaveFileName(KUrl(), QString::null, 
                                   Kopete::UI::Global::mainWidget(), 
                                   QString(i18n("File Transfer '%1' from %2"))
                                   .arg(remote_filename)
                                   .arg(buddy->nickName()));

    if(fileName.isEmpty()) {
      SilcTK::silc_client_file_close(client, conn, session_id);
      return;
    }

    QFileInfo fileInfo(fileName);
    if(fileInfo.exists()) {
      int answer = KMessageBox::questionYesNo
        (Kopete::UI::Global::mainWidget(),
         QString(i18n("The file %1 does already exist. "
                      "Do you want to overwrite it?"))
         .arg(fileInfo.fileName()),
         QString(i18n("File Transfer '%1' from %2"))
         .arg(remote_filename)
         .arg(buddy->nickName()));

      if(answer == KMessageBox::Yes) {
        QFile::remove(fileName);
        break;
      }
    }
    else 
      break;
  }
  completion(fileName.toLatin1().constData(), completion_context);
}

void
SilcFileTransfer::sendFileMonitor(SilcTK::SilcClient,
				  SilcTK::SilcClientConnection,
				  SilcTK::SilcClientMonitorStatus status,
				  SilcTK::SilcClientFileError error,
				  SilcTK::SilcUInt64 offset,
				  SilcTK::SilcUInt64 filesize,
				  SilcTK::SilcClientEntry,
				  SilcTK::SilcUInt32, const char *filepath,
				  void *ctx)
{
  SilcFileTransfer *transfer = (SilcFileTransfer *)ctx;

  if(!transfer) return;

  switch(status) {
  case SilcTK::SILC_CLIENT_FILE_MONITOR_KEY_AGREEMENT:
    std::cerr << "key agreement." << std::endl;
    break;

  case SilcTK::SILC_CLIENT_FILE_MONITOR_SEND:
    std::cerr << "send: " 
	      << (int) offset / filesize * 100 << "%" << std::endl;
    transfer->setOffset(offset);
    break;

  case SilcTK::SILC_CLIENT_FILE_MONITOR_RECEIVE:
    if (! transfer->validTransfer())
      transfer->initTransfer(QString(filepath), (qlonglong)filesize);
    transfer->setOffset(offset);
    std::cerr << "receive." <<std::endl;
    break;

  case SilcTK::SILC_CLIENT_FILE_MONITOR_GET:
    std::cerr << "get." << std::endl;
    break;

  case SilcTK::SILC_CLIENT_FILE_MONITOR_PUT:
    std::cerr << "put." << std::endl;
    break;
  case SilcTK::SILC_CLIENT_FILE_MONITOR_DISCONNECT:
    std::cerr << "disconntected." << std::endl;
    if (! transfer->complete())
      transfer->error(SilcTK::SILC_CLIENT_FILE_ERROR);
    transfer->close();
    break;
  case SilcTK::SILC_CLIENT_FILE_MONITOR_CLOSED:
    std::cerr << "closed." << std::endl;
    transfer->deleteLater();
    break;
  case SilcTK::SILC_CLIENT_FILE_MONITOR_ERROR:
    transfer->error(error);
    transfer->close();
    break;
  }
}

void 
SilcFileTransfer::slotTransferResult()
{
  if(mKopeteTransfer->error() == KIO::ERR_USER_CANCELED) { 
    kDebug() << "Filetransfer aborted" << endl;
    close();
  }
}

#include "silcfiletransfer.moc"
