// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */


#ifndef _CHANNELSTORE_H
#define _CHANNELSTORE_H

#include <qstring.h>
#include <qptrlist.h>
#include <qstringlist.h>
#include <qobject.h>

#include "channelstoreiface.h"
#include "channel.h"
#include "kdetvchannelplugin.h"

class Kdetv;
class QIODevice;

template<class T>
class ChannelPtrList : public QPtrList<T>
{
protected:
	virtual int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2)
    {
        Channel* i1 = static_cast<Channel*>(item1);
        Channel* i2 = static_cast<Channel*>(item2);
        if (i1 && i2)
            if (i1->number() < i2->number()) {
                return -1;
            } else if (i1->number() > i2->number()) {
                return 1;
            }
        return 0;
    }
};

typedef ChannelPtrList<Channel> ChannelList;

/**
 * A suite of channels.
 */
class LIBKDETV_EXPORT ChannelStore : public QObject, public virtual ChannelStoreIface 
{
    Q_OBJECT
public:
    ChannelStore( Kdetv *ktv, QObject *parent, const char *name=0);
    virtual ~ChannelStore();

    //ChannelList *channels() { return &_channels; }

    uint count() const { return _channels.count(); }
    bool isEmpty() const { return count() ? false : true; }

    /**
     * Load and save channels.
     * If no format is supplied, use autodetection (this is only available for loading a file)
     * If no format for saving is supplied, the default format is used.
     */
    bool load( const QString &filename, const QString& fmt=QString::null );
    bool load( QIODevice* dev, const QString& fmt );
    bool save( const QString &filename, const QString& fmt="xml" );
    bool save( QIODevice* dev, const QString& fmt="xml" );

    /**
     * Returns default channel storage file name
     */
    const QString& defaultFileName() const { return _defaultFileName; };

    /*
     * Provides mappings between user readable and internal format names.
     * Key is user readable, value is internal name.
     * Internal name can be supplied to load/save functions as format argument.
     */
    const QMap<QString, QString>& fileFormatsRead()  const;
    const QMap<QString, QString>& fileFormatsWrite() const;

    /**
     * Returns default file format
     * It's safe to return a hardcoded string since this is our "inhouse" format that is always there
     */
    const QString& defaultFormat() const { static QString f = "xml"; return f; };

    Channel *channelAt( int idx ) { return _channels.at(idx); }
    Channel *channelNumber( int n );

    Channel *channelAfter( Channel *channel );
    Channel *channelBefore( Channel *channel );

    KdetvChannelPlugin::ChannelFileMetaInfo& getMetaInfo() const { return *_metaInfo; }     

public slots:
    void addChannel( Channel *channel );

    int removeChannel(int idx);
    int removeChannelNumber(int n);
    int removeChannel(Channel *channel);

    /** Adds channels from @p nstore to the end of this ChannelStore. */
    void addChannels( const ChannelStore& nstore);
    void clear();

    /** Renumbers the channels starting at supplied number */
    void renumber( int start );

    /** Renumbers the channels starting at default */
    void renumber();

signals:
    void channelAdded(Channel *x);
    void channelRemoved(Channel *x);
    void aboutToReload();
    void loaded();
    void saved();

private:
    // This enables loading of old channel files without loosing control values
    void fixupControlLists();

    ChannelList _channels;
    bool _silentModifications;
    Kdetv *_ktv;
    QString _defaultFileName;
    KdetvChannelPlugin::ChannelFileMetaInfo* _metaInfo;
};


#endif
