/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
 * These codes originated in gtkhpaned.c and gtkvpaned.c in GTK+.
 */

#include "kz-paned.h"

#include "gtk-utils.h"
#include <glib/gi18n.h>

enum {
	CHANGED_POSITION,
	LAST_SIGNAL
};

/* object class */
static void size_request   (GtkWidget      *widget,
                            GtkRequisition *requisition);
static void size_allocate  (GtkWidget      *widget,
                            GtkAllocation  *allocation);

static void kz_paned_rotate_position      (KzPaned *kzpaned);
static void kz_paned_flip_position        (KzPaned *kzpaned);
static void kz_paned_set_divider_position (KzPaned *kzpaned);

#define KZ_PANED_SET_VERTICAL(kzpaned)					\
{									\
	GdkDisplay *display;						\
	GdkCursorType cursor_type = GDK_SB_V_DOUBLE_ARROW;		\
									\
	GTK_PANED(kzpaned)->cursor_type = cursor_type;			\
	if (GTK_PANED(kzpaned)->handle)					\
	{								\
		GdkCursor *cur;						\
		display = gtk_widget_get_display(GTK_WIDGET(kzpaned));	\
		cur = gdk_cursor_new_for_display(display, cursor_type);	\
		gdk_window_set_cursor(GTK_PANED(kzpaned)->handle, cur);	\
		gdk_cursor_unref(cur);					\
	}								\
	GTK_PANED(kzpaned)->orientation = GTK_ORIENTATION_HORIZONTAL;	\
}

#define KZ_PANED_SET_HORIZONTAL(kzpaned)				\
{									\
	GdkDisplay *display;						\
	GdkCursorType cursor_type = GDK_SB_H_DOUBLE_ARROW;		\
									\
	GTK_PANED(kzpaned)->cursor_type = cursor_type;			\
	if (GTK_PANED(kzpaned)->handle)					\
	{								\
		GdkCursor *cur;						\
		display = gtk_widget_get_display(GTK_WIDGET(kzpaned));	\
		cur = gdk_cursor_new_for_display(display, cursor_type);	\
		gdk_window_set_cursor(GTK_PANED(kzpaned)->handle, cur);	\
		gdk_cursor_unref(cur);					\
	}								\
	GTK_PANED(kzpaned)->orientation = GTK_ORIENTATION_VERTICAL;	\
}

static gint kz_paned_signals[LAST_SIGNAL] = {0};

static void (* kz_paned_hsize_request)  (GtkWidget       *widget,
					 GtkRequisition  *requision);
static void (* kz_paned_vsize_request)  (GtkWidget       *widget,
					 GtkRequisition  *requision);
static void (* kz_paned_hsize_allocate) (GtkWidget       *widget,
					 GtkAllocation   *alocation);
static void (* kz_paned_vsize_allocate) (GtkWidget       *widget,
					 GtkAllocation   *alocation);

G_DEFINE_TYPE(KzPaned, kz_paned, GTK_TYPE_PANED)

static void
kz_paned_class_init (KzPanedClass *klass)
{
	GtkWidgetClass *widget_class;
	GtkHPanedClass *hpaned_class;
	GtkVPanedClass *vpaned_class;

	gtk_hpaned_new();
	gtk_vpaned_new();
	
	hpaned_class = g_type_class_peek (GTK_TYPE_HPANED);
	vpaned_class = g_type_class_peek (GTK_TYPE_VPANED);

	kz_paned_hsize_request =
		((GtkWidgetClass *)hpaned_class)->size_request;
	kz_paned_vsize_request =
		((GtkWidgetClass *)vpaned_class)->size_request;
	kz_paned_hsize_allocate =
		((GtkWidgetClass *)hpaned_class)->size_allocate;
	kz_paned_vsize_allocate =
		((GtkWidgetClass *)vpaned_class)->size_allocate;

	widget_class  = (GtkWidgetClass *) klass;

	widget_class->size_request  = size_request;
	widget_class->size_allocate = size_allocate;
	
	klass->changed_position = NULL;

	kz_paned_signals[CHANGED_POSITION]
		= g_signal_new ("changed_position",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzPanedClass, changed_position),
				NULL, NULL,
				g_cclosure_marshal_VOID__INT,
				G_TYPE_NONE, 1,
				G_TYPE_INT);
}


static void
kz_paned_init (KzPaned *kzpaned)
{
	KZ_PANED_SET_HORIZONTAL(kzpaned);	
	kzpaned->position  = GTK_POS_LEFT;
}


GtkWidget *
kz_paned_new (void)
{
	KzPaned *kzpaned;

	kzpaned = g_object_new (KZ_TYPE_PANED, NULL);

	return GTK_WIDGET (kzpaned);
}

gboolean
kz_paned_is_showing_all_children (KzPaned *kzpaned)
{
	GtkPaned *paned;

	paned = GTK_PANED(kzpaned);

	return paned->child1 && GTK_WIDGET_VISIBLE(paned->child1) &&
		paned->child2 && GTK_WIDGET_VISIBLE(paned->child2);
}

void
kz_paned_set_separator_position (KzPaned *kzpaned, gint position)
{
	GtkPaned *paned;
	GtkWidget *child1;
	gint child1_width, child1_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child1_width  = child1->allocation.width;
	child1_height = child1->allocation.height;

	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_TOP:
		gtk_paned_set_position(paned, position);
		break;
	 case GTK_POS_RIGHT:
		gtk_paned_set_position(paned, child1_width - position);
		break;
	 case GTK_POS_BOTTOM:
		gtk_paned_set_position(paned, child1_height - position);
		break;
	 default:
		break;
	}
}

void
kz_paned_set_position(KzPaned *kzpaned, GtkPositionType position)
{
	GtkPositionType prev_pos;
	prev_pos = kzpaned->position;

	kzpaned->position = position;
	switch(position)
	{
	 case GTK_POS_LEFT:
		KZ_PANED_SET_HORIZONTAL(kzpaned);
		if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_RIGHT)
			kz_paned_flip_position(kzpaned);
		break;
	 case GTK_POS_RIGHT:
		KZ_PANED_SET_HORIZONTAL(kzpaned);
		if (prev_pos == GTK_POS_TOP)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_LEFT)
			kz_paned_flip_position(kzpaned);
		else if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_set_divider_position(kzpaned);
		break;
	 case GTK_POS_TOP:
		KZ_PANED_SET_VERTICAL(kzpaned);
		if (prev_pos == GTK_POS_RIGHT)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_flip_position(kzpaned);
		break;
	 case GTK_POS_BOTTOM:
		KZ_PANED_SET_VERTICAL(kzpaned);
		if (prev_pos == GTK_POS_LEFT)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_TOP)
			kz_paned_flip_position(kzpaned);
		else if (prev_pos == GTK_POS_RIGHT)
			kz_paned_set_divider_position(kzpaned);
		break;
	 default:
		break;
	}	
	gtk_widget_queue_resize(GTK_WIDGET(kzpaned));

	g_signal_emit(kzpaned,
		      kz_paned_signals[CHANGED_POSITION],
		      0,
		      position);
}


static void
kz_paned_set_divider_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child1_width, child1_height;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child1_width  = child1->allocation.width;
	child1_height = child1->allocation.height;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	switch(kzpaned->position)
	{
	 case GTK_POS_RIGHT:/* only from GTK_POS_BOTTOM */
		gtk_paned_set_position(paned, child1_width - child2_height);
		break;
	 case GTK_POS_BOTTOM: /* only from GTK_POS_RIGHT */
		gtk_paned_set_position(paned, child1_height - child2_width);
		break;
	 default:
		break;
	}
}


static void
kz_paned_rotate_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child1_width, child1_height;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child1_width  = child1->allocation.width;
	child1_height = child1->allocation.height;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	paned->child1 = child2;
	paned->child2 = child1;
	
	/*
	 * When pane is divided in horizontal, child1_height always equals child2_height.
	 * When pane is divided in vertical, child1_width always equals child2_width.
	 */

	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT: /* only from GTK_POS_BOTTOM */
		gtk_paned_set_position(paned, child2_height);
		break;
	 case GTK_POS_RIGHT:/* only from GTK_POS_TOP */
		gtk_paned_set_position(paned, child1_width - child1_height);
		break;
	 case GTK_POS_TOP: /* only from GTK_POS_RIGHT */
		gtk_paned_set_position(paned, child2_width);
		break;
	 case GTK_POS_BOTTOM: /* only from GTK_POS_LEFT */
		gtk_paned_set_position(paned, child1_height - child1_width);
		break;
	 default:
		break;
	}
}


static void
kz_paned_flip_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	paned->child1 = child2;
	paned->child2 = child1;
	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		gtk_paned_set_position(paned, child2_width);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		gtk_paned_set_position(paned, child2_height);
		break;
	 default:
		break;
	}
}


static void
size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	KzPaned *kzpaned;
	g_return_if_fail (KZ_IS_PANED (widget));
	kzpaned = KZ_PANED (widget);

	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		kz_paned_hsize_request(widget, requisition);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		kz_paned_vsize_request(widget, requisition);
		break;
	 default:
		break;
	}
}


static void
size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	KzPaned *kzpaned;
	g_return_if_fail (KZ_IS_PANED (widget));
	kzpaned = KZ_PANED (widget);
	
	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		kz_paned_hsize_allocate(widget, allocation);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		kz_paned_vsize_allocate(widget, allocation);
		break;
	 default:
		break;
	}
}
