/*
    kcheckdirectoryselector.h - Show a treeview of all directories with a checkbox.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    Based on code in amarok/directorylist.[h/cpp] by
    (C) 2003 Scott Wheeler <wheeler@kde.org>
    (C) 2004 Max Howell <max.howell@methylblue.com>
    (C) 2004 Mark Kretschmann <markey@web.de>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#ifndef KAMEFU_UIKCHECKDIRECTORYSELECTOR_H
#define KAMEFU_UIKCHECKDIRECTORYSELECTOR_H

// Qt includes
#include <qcheckbox.h>
#include <qlistview.h>

#include <kfileitem.h>

class KURL;
class KDirLister;
class QPainter;
class QColorGroup;

namespace Kamefu {

namespace UI {

/**
 * @brief Show a treeview of all directories with a checkbox. 
 * This a pretty and useful widget that show a list of all directories on the system.
 * Mutliple directories can be checked. Directories can be recursed if the user want it. 
 *
 * Using this widget is pretty simple
 * @code
	KCheckDirectorySelector *selector = new KCheckDirectorySelector(this, "selector", true);
	// Retrive selected directories
	QStringList directories  = selector->dirs();
 * @endcode
 *
 * @author Scott Wheeler <wheeler@kde.org>
 * @author Max Howell <max.howell@methylblue.com>
 * @author Mark Kretschmann <markey@web.de>
 * @author Michaël Larouche <michael.larouche@kdemail.net>
 *
 */
class KCheckDirectorySelector : public QListView
{
Q_OBJECT
public:
	/**
	 * Create a new KCheckDirectorySelector.
	 * @param parent Parent Widget
	 * @param name Widget name
	 * @param recursive Does a item recursive into his childrens ? False by default.
	 */
	KCheckDirectorySelector(QWidget *parent = 0, const char *name = 0, bool recursive = false);
	virtual ~KCheckDirectorySelector();

	/**
	 * Reimplement sizeHint to have directoryList not being too big for "low" resolutions.
	 */
	QSize sizeHint() const
	{
		return QSize(400, 100);
	}

	/**
	 * @brief Return the list of selected directories. 
	 * Not returning const because this list can be modified by KCheckDirectorySelectorItem.
	 */ 
	QStringList& dirs();
	/**
	 * Set a predefined list of selected directories. Usefull when restoring settings.
	 * @param dirs Predefined list of selected directories.
	 */
	void setDirs(const QStringList &dirs);

	// FIXME: Maybe make then private and make KCheckDirectorySelectorItem friend.
	void addDirectory(const QString &directory);
	void removeDirectory(const QStringList::Iterator &directory);

	/**
	 * Does a item recursive into his childrens ?
	 */ 
	bool recursive() const;
	/**
	 * Enable or disable recursivity.
	 * @param recursive Set recursivity
	 */
	void setRecursive(bool recursive);

private:
	class Private;
	Private *d;
};

/**
 * @internal
 * @brief Item of a KCheckDirectorySelector.
 * This class inherit from QCheckListItem.
 * It does the real listing of the directories. It also paint the item in a different color if the item is active.
 *
 * @author Scott Wheeler <wheeler@kde.org>
 * @author Max Howell <max.howell@methylblue.com>
 * @author Mark Kretschmann <markey@web.de>
 * @author Michaël Larouche <michael.larouche@kdemail.net>
 *
 */
class KCheckDirectorySelectorItem : public QObject, public QCheckListItem
{
Q_OBJECT
public:
	KCheckDirectorySelectorItem( KCheckDirectorySelector *parent );
	KCheckDirectorySelectorItem( QListViewItem *parent, const KURL &url );
	~KCheckDirectorySelectorItem();

	QCheckListItem *parent() const { return (QCheckListItem*)QListViewItem::parent(); }
	bool isDisabled() const;
	QString fullPath() const;

	void setOpen(bool b); // reimpl.
	void stateChange(bool); // reimpl.
	void activate(); // reimpl.
	void paintCell(QPainter * p, const QColorGroup & cg, int column, int width, int align); // reimpl.

public slots:
	void newItems( const KFileItemList& );
	void completed() { if( childCount() == 0 ) { setExpandable( false ); repaint(); } }

private:
	class Private;
	Private *d;
};

}

}

#endif
