/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *,libcmt.lib,libc.lib,
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>

#ifdef _WIN32
 #include <sys/stat.h>
 #include <windows.h>

 #ifndef S_ISREG
 #define S_ISREG(mode)  (((mode) & S_IFMT) == S_IFREG)
 #endif

 #ifndef FTW_F
 #define FTW_F 1
 #endif
#else
 #include <ftw.h>
 #include <dirent.h>
 #include <assert.h>
#endif

/* From taglib*/
#include <taglib/tag.h>
#include <taglib/fileref.h>

/* Consts and variables for Files and Dirs */
char start_dir[255];
int  be_verbose ;
int  add_empty ;
int  sort_files ;
bool use_utf8 ;
bool cap_entries ;

/* Consts and variables for writing the DB */
const char DB_file[] = "iRivNavi.iDB";
FILE* DB_handle;
const int FAT_start = 0x0080,
      head_start = 0x0040,
      tags_start = 0xA0C0;
int number_of_tags;
int last_tag_offset;
char** files ;


int deslash ( char * string )
{
  unsigned count;

  for ( count=0 ; count < strlen ( string ) ; count ++)
  {
    if (string[count] == '/')
    {
        string[count] = '\\';
    }
  }
  return 0;
}


void str_ntoup ( char* str , int limit )
{
  char* this_char;

  this_char = str ;

  while ( (*this_char != 0x00) && ( this_char - str < limit )  )
  {
    *this_char = toupper ( *this_char );
    this_char++;
  }
}

/*
 * Capitalize the first letter after
 * ' ', '_', '.', ',', '-', ':', ';', '(', ')', '&' and numbers
 */
void str_capfirst ( char* str )
{
  char* this_char;
  char* previous_char = NULL;

  this_char = str ;

  while ( (*this_char != 0x00) )
  {
    if ( previous_char==NULL || *previous_char==' ' || *previous_char=='_' ||
         *previous_char=='.' || *previous_char==',' || *previous_char=='-' ||
         *previous_char==':' || *previous_char==';' || *previous_char=='(' ||
         *previous_char==')' || *previous_char=='&' || *previous_char=='1' ||
         *previous_char=='2' || *previous_char=='3' || *previous_char=='4' ||
         *previous_char=='5' || *previous_char=='6' || *previous_char=='7' ||
         *previous_char=='8' || *previous_char=='9' || *previous_char=='0')
      *this_char = toupper ( *this_char );
    else
      *this_char = tolower ( *this_char );

    previous_char = this_char;
    this_char++;
  }
}

/*
 * If the string is null return "Unknown"
 * otherwise remove the spaces from the head
 * of the string and its tail, then copy it into a char[].
 * You have to free the space allocated by this
 * string with "free".
 */
char* prepare_playlist_entry(std::string entry)
{
  char* result=NULL;
  std::string work = entry;

  // remove space from the head
  unsigned endpos = work.find_first_not_of(' ', 0);
  if ( endpos > 0 && endpos < work.length() )
    work = work.erase(0, endpos - 1);
  // remove space from the tail
  unsigned startpos = work.find_last_not_of(' ', work.length() - 1);
  if ( !work.empty() && startpos < (work.length() - 1) )
    work = work.erase(startpos + 1, work.length() - 1);

  if ( work.empty() )
    work = "Unknown";

  // copy the string into a char array.
  result = (char *)realloc(result, work.length() + 1);
  strcpy(result, work.c_str());

  return result;
}

void get_lso16 ( char bytes[2], int number )
{
  bytes[0] = (unsigned char) number;
  number >>= 8 ;
  bytes[1] = (unsigned char) number;
}


void get_lso32 ( char bytes[4], int number)
{
  bytes[0] = (unsigned char) number;
  number >>= 8 ;
  bytes[1] = (unsigned char) number;
  number >>= 8 ;
  bytes[2] = (unsigned char) number;
  number >>= 8 ;
  bytes[3] = (unsigned char) number;
}


int startup_DB ( void )
{
  int byte_count = 0 ;
  char full_name[255];

  number_of_tags = 0;

  strcpy ( full_name , start_dir );
  strcat ( full_name , "/" );
  strcat ( full_name , DB_file );
  DB_handle = fopen ( full_name , "wb" );

  /* Check file was created */
  if (DB_handle == 0x00)
  {
    printf("Could not create DB file - check that the supplied path exists and is writeable\n");
    exit(1);
  }

  /* Ok, first we write a clean Header and FAT*/
  for ( byte_count = 0 ; byte_count < tags_start ; byte_count++ )
  {
    fputc ( 0 , DB_handle );
  }

  /* now we'll write some marks */
  fseek ( DB_handle , 0x0000 , SEEK_SET );
  fputs ( "iRivDB Ver 0.12" , DB_handle );
  fseek ( DB_handle , 0x0020 , SEEK_SET );
  fputs ( "iRiver iHP-100 DB File" , DB_handle );
  fseek ( DB_handle , 0xA080 , SEEK_SET );
  fputs ( "Designed by iRiver" , DB_handle );

  return 0;
}


int update_DB ( const char *filename,
                const char *songtitle,
                const char *artist,
                const char *album,
                const char *genre )
{
  int fn_lng,st_lng,at_lng,al_lng,gn_lng;
  int previous_tag_offset;
  char  new_filename[255] = "";
  char lso16[2];
  char lso32[4];

  previous_tag_offset = last_tag_offset;

  strcat ( new_filename , filename + strlen ( start_dir ) );
  deslash ( new_filename );

  fn_lng = strlen ( new_filename ) + 1 ;
  st_lng = strlen ( songtitle ) + 1 ;
  at_lng = strlen ( artist ) + 1;
  al_lng = strlen ( album ) + 1 ;
  gn_lng = strlen ( genre ) + 1 ;


  fseek ( DB_handle , tags_start + last_tag_offset , SEEK_SET );
  get_lso16 ( lso16 , fn_lng ) ;
  fputc ( lso16[0] , DB_handle ) ;
  fputc ( lso16[1] , DB_handle ) ;
  get_lso16 ( lso16 , st_lng ) ;
  fputc ( lso16[0] , DB_handle ) ;
  fputc ( lso16[1] , DB_handle ) ;
  get_lso16 ( lso16 , at_lng ) ;
  fputc ( lso16[0] , DB_handle ) ;
  fputc ( lso16[1] , DB_handle ) ;
  get_lso16 ( lso16 , al_lng ) ;
  fputc ( lso16[0] , DB_handle ) ;
  fputc ( lso16[1] , DB_handle ) ;
  get_lso16 ( lso16 , gn_lng ) ;
  fputc ( lso16[0] , DB_handle ) ;
  fputc ( lso16[1] , DB_handle ) ;
  last_tag_offset += 10;

  fseek ( DB_handle , tags_start + last_tag_offset , SEEK_SET );
  fputs ( new_filename , DB_handle );
  fputc ( 0 , DB_handle );
  fputs ( songtitle , DB_handle );
  fputc ( 0 , DB_handle );
  fputs ( artist , DB_handle );
  fputc ( 0 , DB_handle );
  fputs ( album , DB_handle );
  fputc ( 0 , DB_handle );
  fputs ( genre , DB_handle );
  fputc ( 0 , DB_handle );
  last_tag_offset += fn_lng + st_lng + at_lng + al_lng + gn_lng ;

  fseek ( DB_handle , FAT_start + 4 * number_of_tags , SEEK_SET ) ;
  get_lso32 ( lso32 , previous_tag_offset ) ;
  fputc ( lso32[0] , DB_handle ) ;
  fputc ( lso32[1] , DB_handle ) ;
  fputc ( lso32[2] , DB_handle ) ;
  fputc ( lso32[3] , DB_handle ) ;


  number_of_tags++;
  fseek ( DB_handle , head_start , SEEK_SET ) ;
  get_lso32 ( lso32 , number_of_tags ) ;
  fputc ( lso32[0] , DB_handle ) ;
  fputc ( lso32[1] , DB_handle ) ;
  fputc ( lso32[2] , DB_handle ) ;
  fputc ( lso32[3] , DB_handle ) ;

  return 0;
}


int close_DB ( void )
{
  return fclose ( DB_handle );
}


int process_File ( const char* filename )
{
  char *f_album=NULL,
       *f_artist=NULL,
       *f_songtitle=NULL,
       *f_genre=NULL;
  TagLib::Tag *t;


  if (be_verbose) printf ( "Processing file: %s\n" , filename ) ;

  TagLib::FileRef f( filename );
  if(f.isNull())
  {
    if (be_verbose) printf ( "Warning: File [%s] could not be opened.\n", filename );
    return 1;
  }

  t = f.tag();

  if ( !add_empty && t->isEmpty() )
  {
    if (be_verbose) printf ( "Warning: File [%s] has a no valid tag, not added.\n", filename );
    return 1;
  }

  /* TITLE */
  f_songtitle = prepare_playlist_entry( t->title().to8Bit(use_utf8) ) ;
  if (cap_entries && !use_utf8) str_capfirst( f_songtitle ) ;
  if (be_verbose) printf ( "Songtitle: %s\n" , f_songtitle ) ;

  /* ARTIST */
  f_artist = prepare_playlist_entry( t->artist().to8Bit(use_utf8) ) ;
  if (cap_entries && !use_utf8) str_capfirst( f_artist ) ;
  if (be_verbose) printf ( "Artist: %s\n" , f_artist ) ;

  /* ALBUM */
  f_album = prepare_playlist_entry( t->album().to8Bit(use_utf8) ) ;
  if (cap_entries && !use_utf8) str_capfirst( f_album ) ;
  if (be_verbose) printf ( "Album: %s\n" , f_album ) ;

  /* GENRE */
  f_genre = prepare_playlist_entry( t->genre().to8Bit(use_utf8) ) ;
  if (cap_entries && !use_utf8) str_capfirst( f_genre ) ;
  if (be_verbose) printf ( "Genre: %s\n" , f_genre ) ;

  if (be_verbose) printf ( "---------------------------------------------------\n") ;


  update_DB ( filename ,
              f_songtitle ,
              f_artist ,
              f_album ,
              f_genre ) ;

  if (f_songtitle) free(f_songtitle);
  if (f_artist) free(f_artist);
  if (f_album) free(f_album);
  if (f_genre) free(f_genre);

  return 0;
}


int process_file( char* filename, const struct stat* filestat, int file_type)
{
  char ext[4];
  //TagLib::File *f;

  if( (file_type == FTW_F) && S_ISREG(filestat->st_mode) )
  {
    strncpy ( ext , filename + strlen(filename) - 4 , 4 );
    str_ntoup ( ext , 4 );
    if ( !strncmp ( ext , ".MP3" , 4 ) || !strncmp ( ext , ".OGG" , 4 ) )
    {
      process_File ( filename ) ;
    }
    else
    {
      if (be_verbose)
        printf ( "Warning: File [%s] - File format not supported.\n", filename );
    }
  }
  return 0;
}



#ifdef _WIN32

typedef void (*execfn)(char*, struct stat*, int file_type);

void win32_recurse_dir(char* path, execfn before)
{
    /* In depth traversal of the subdir tree */
  WIN32_FIND_DATA find_file_data;
  HANDLE hnd;

  struct stat stat_buf; /* We have to have one local because of after */

  int path_len = strlen(path);

  /* current node */
  if (stat(path, &stat_buf))
    perror(path);

  /* execute before for the current node */
  if (before)
  {
    (*before)(path, &stat_buf, FTW_F);
  }

  /* if it is a directory, recurse through all sons */
  if ((stat_buf.st_mode & S_IFMT) == S_IFDIR)
  {
    strcat(path, "/*");
    hnd = FindFirstFile(path, &find_file_data);

    while (hnd != INVALID_HANDLE_VALUE && FindNextFile(hnd, &find_file_data) != FALSE)
  {
      if(!strcmp(find_file_data.cFileName, ".") ||
     !strcmp(find_file_data.cFileName, "..")) continue;

      path[path_len+1] = '\0';
      strcat(path, find_file_data.cFileName);
      win32_recurse_dir(path, before);
    }
    path[path_len] = '\0';
    FindClose(hnd);
  }
}

#else

static int compare_fn(const void *a, const void *b)
{
  return strcoll(*(const char **) a, *(const char **) b);
}

static void sorted_recurse(char* dirname)
{
  struct stat    statbuf;
  struct dirent* thisent;
  DIR*           thisdir = opendir(dirname);
  char*          sub_name = NULL;
  char**         files = NULL;
  size_t         dirname_l, strsz;
  int            n_files = 0, i;

  if (!thisdir)
    return;
  if (be_verbose) printf ( "Sorting directory: %s\n" , dirname ) ;
  dirname_l = strlen(dirname);
  while ( (thisent = readdir(thisdir)) )
  {
    if ((strcmp(thisent->d_name, ".") == 0) ||
      (strcmp(thisent->d_name, "..") == 0))
      continue;

    strsz = dirname_l + strlen(thisent->d_name) + 2;
    sub_name = (char *)realloc(sub_name, strsz);
    assert(sub_name);
    sprintf(sub_name, "%s/%s", dirname, thisent->d_name);
    assert( !stat(sub_name, &statbuf) ) ;
    if ( S_ISDIR(statbuf.st_mode) )
    {
      sorted_recurse(sub_name);
    }
    else if ( S_ISREG(statbuf.st_mode) )
    {
      n_files++;
      files = (char **)realloc(files, n_files * sizeof(files[0]));
      assert(files);
      files[n_files - 1] = strdup(thisent->d_name);
      assert(files[n_files - 1]);
    }
  }
  closedir(thisdir);

  if (n_files)
  {
    qsort(files, n_files, sizeof(files[0]), compare_fn);
    for (i = 0; i < n_files; i++)
    {
      strsz = dirname_l + strlen(files[i]) + 2;
      sub_name = (char *)realloc(sub_name, strsz);
      assert(sub_name);
      sprintf(sub_name, "%s/%s", dirname, files[i]);
      if (stat(sub_name, &statbuf) == 0)
        process_file(sub_name, &statbuf, FTW_F);
      free(files[i]);
    }
    free(files);
  }
  if (sub_name)
    free(sub_name);

  return;
}

#endif


int recurse_dir( char* dirname )
{
#ifdef _WIN32
  char temp_buffer[sizeof(start_dir)];
  strcpy(temp_buffer, dirname);
  win32_recurse_dir(temp_buffer, &process_file);
#else
  if (sort_files)
    sorted_recurse(dirname);
  else
    ftw( dirname, (__ftw_func_t) &process_file, 30 );
#endif
  return 0;
}


void print_help ( void )
{
  printf ( "iRipDB v0.1.3b - Create DB file for iRiver iHP players\n" ) ;
  printf ( "Usage: iripdb [-vhe] path\n\n" ) ;
  printf ( "\t-v\tverbose operation\n" ) ;
  printf ( "\t-h\tprint this help information\n" ) ;
  printf ( "\t-e\tadd also files not tagged (with generic tags)\n" ) ;
  printf ( "\t-s\tsort files in directory\n" ) ;
  printf ( "\t-u\tuse UTF-8 for encoding of the entries\n" ) ;
  printf ( "\t-c\tcapitalize the first word letter in db entries (only works for ASCII)\n" ) ;
  printf ( "\tpath\tpath to the iHP's mount point\n" ) ;
}


void print_usage ( void )
{
  printf ( "Usage: iripdb [-vhesuc] path_to_ihp\n" );
}


int main ( int argc, char **argv )
{
  int argpnt;
  unsigned charpnt;

  *start_dir = 0x00 ;
  /* By default it will be verbose-less */
  be_verbose = 0 ;
  /* By default it won't add empty files */
  add_empty = 0 ;
  /* By default don't sort files after their filename */
  sort_files = 0 ;
  /* By default use ansi encoding */
  use_utf8 = false ;
  /* By default don't capitalize the db entries */
  cap_entries = false ;



  /* Process arguments */
  for ( argpnt = 1 ; argpnt < argc ; argpnt++ )
  {
    if ( *argv[argpnt] == '-' )
    {
      for ( charpnt = 1 ; charpnt < strlen ( argv[argpnt] ) ; charpnt++ )
      {
        switch ( argv[argpnt][charpnt] )
        {
          case  'v':
            be_verbose = 1 ;
            break;
          case  'e':
            add_empty = 1 ;
            break;
          case  's':
            sort_files = 1 ;
            break;
          case  'u':
            use_utf8 = true ;
            break;
          case  'c':
            cap_entries = true ;
            break;
          case  'h':
            print_help () ;
            return 0 ;
            break;
          default:
            printf ( "error: invalid argument -%c.\n" ,
              argv[argpnt][charpnt] ) ;
            print_usage () ;
            return 1 ;
            break;
        }

      }
    }
    else
    {
      strcpy ( start_dir , argv[argpnt] );
    }
  }

  if ( *start_dir != 0x00 )
  {
    if ( start_dir[strlen(start_dir)-1] == '/' ) {
      start_dir[strlen(start_dir)-1] = 0x00 ;
    }
    if (be_verbose) printf( "Building up the iRivDB in current dir.\n" );
    if (be_verbose) printf( "Writing header information.\n");
    startup_DB ();
    if (be_verbose) printf( "Start directory recursion.\n");
    recurse_dir ( start_dir );
    if (be_verbose) printf( "Closing database file.\n");
    close_DB ();
    printf ( " %i file(s) added at %s/%s\n" , number_of_tags ,
          start_dir , DB_file );
    return (0);
  }
  else
  {
    printf ( "error: no path provided\n" );
    print_usage () ;
    return 1;
  }
}
