/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ibasicscript.h"


#include "iscriptkit.h"

//
//  Templates
//
#include "icalculatortemplate.h"
#include "iscriptkittemplate.h"


namespace iBasicScript_Private
{
	template<class T>
	class Declaration : public iScriptKit::VariableDeclarationStatement
	{

	public:

		Declaration(iScript *script, const iString &command) : iScriptKit::VariableDeclarationStatement(script,command)
		{
		}

	protected:

		virtual iScriptKit::Prototype* CreatePrototype(const iString &name, int dim) const
		{
			return this->iScriptKit::VariableDeclarationStatement::NewPrototype<T>(name,dim);  //  full specification is needed for some compilers
		}
	};

		
	class Loop1 : public iScriptKit::LoopFlowControlStatement
	{

	public:

		Loop1(iScript *script, const iString &command) : iScriptKit::LoopFlowControlStatement(script,command,false)
		{
		}

	protected:

		virtual bool ParseOperandForCount()
		{
			//
			//  Do nothing - the whole argument is the count expression
			//
			return true;
		}
	};


	class Loop2 : public iScriptKit::LoopFlowControlStatement
	{

	public:

		Loop2(iScript *script, const iString &command) : iScriptKit::LoopFlowControlStatement(script,command,true)
		{
			mIndexVariable = 0;
		}

	protected:

		virtual bool ParseOperandForCount()
		{
			//
			//  Look for the variable name in the context:
			//  (for) var to N  (old syntax)
			//  (for) var=b,e,s
			//  (for) var = b,e,s
			//
			iString s(mOperand);
			s.ReduceWhiteSpace();
			iString vn = s.Section(" ",0,0);
			if(vn.Contains('=')) vn = s.Section("=",0,0);

			//
			//  Is this a defined integer variable?
			//
			iScriptKit::Value *var = this->FindVariable(vn);
			if(var == 0)
			{
				this->ReportError("The loop index variable is not declared.");
				return false;
			}

			if(var->GetTypeId()!=0 && var->Dim()!=1)
			{
				this->ReportError("The loop index variable must be an integer scalar.");
				return false;
			}

			mIndexVariable = var;

#ifdef ISCRIPT_BACKWARD_COMPATIBLE
			//
			//  Variable is fine. Is it an old-style syntax?
			//
			if(s.Section(" ",1,1) == "to")
			{
				s = s.Section(" ",2,2);
				mOperand = s;
				return true;
			}
#endif

			//
			//  This is a new-style syntax
			//
			s = mOperand.Section("=",1);
			mFirstExpression = s.Section(",",0,0);
			mOperand = s.Section(",",1,1);
			mStepExpression = s.Section(",",2);
			return true;
		}

		virtual void OnExecute()
		{
			iScriptKit::Variable<int> *var;
			if(mIndexVariable==0 || (var = static_cast<iScriptKit::Variable<int>*>(mIndexVariable))==0)
			{
				this->ReportError("A bug in iBasicScript class is detected.");
			}
			else
			{
				var->Assign(mIndex,0);
			}
		}

		iScriptKit::Value *mIndexVariable;	
	};


	class Conditional : public iScriptKit::ConditionalFlowControlStatement
	{

	public:

		Conditional(iScript *script, const iString &command) : iScriptKit::ConditionalFlowControlStatement(script,command)
		{
		}

	protected:

		virtual bool ParseOperandForCondition()
		{
			//
			//  Look for the argument in the context:
			//  (if) <condition> then
			//
			iString s(mOperand);
			s.ReduceWhiteSpace();
			
			if(s.Part(-4) != "then")
			{
				this->ReportError("Syntax error: if statement must end with 'then'.");
				return false;
			}

			mOperand.Replace("then","");
			return true;
		}
	};
};


using namespace iBasicScript_Private;


iBasicScript::iBasicScript(iScript *parent) : iScript(parent)
{
	//
	//  IFrIT script are case-sensitive since the keys are
	//
	mCaseSensitive = true;

	//
	//  This is for embeding other scripts
	//
	mHeadOfLineContinuationString = "> ";
	mAppendSeparator = "%%";

	//
	//  Dummy words
	//
	this->CreateDummyWord("set");

	//
	//  Aliases
	//
	this->CreateAliasWord("real","float"); 
	this->CreateAliasWord("endif","end"); 

	//
	//  Parameter words
	//
	this->AddConstant(new iScriptKit::Constant<bool>(this,"false",false));
	this->AddConstant(new iScriptKit::Constant<bool>(this,"true",true));

	//
	//  Declarations
	//
	this->AddPrototype< Declaration<int> >("int");
	this->AddPrototype< Declaration<bool> >("bool");
	this->AddPrototype< Declaration<float> >("float");
	this->AddPrototype< Declaration<double> >("double");

#ifdef ISCRIPT_BACKWARD_COMPATIBLE
	this->AddPrototype< Declaration<int> >("var int");
	this->AddPrototype< Declaration<bool> >("var bool");
	this->AddPrototype< Declaration<float> >("var float");
	this->AddPrototype< Declaration<double> >("var double");
#endif

	//
	//  Flow control operations
	//
	this->AddPrototype< Loop1 >("loop");
	this->AddPrototype< Loop2 >("for");
	this->AddPrototype< iScriptKit::ClosingFlowControlStatement >("end");
	this->AddPrototype< iScriptKit::SwappingFlowControlStatement >("else");
	this->AddPrototype< Conditional >("if");
}

