/*
  Copyright 2005, 2006, 2007 David Cade, Damien Stehle.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehle, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/


#ifndef NR_H
#define NR_H

#include <mpfr.h>

#include <iostream>
#include <cstdlib>
#include <cmath>


using namespace std;



template<class Z>
class Z_NR
{
  Z data;
 public:
  Z_NR<Z>();
  ~Z_NR<Z>();
  inline void print();
  inline void printerr();
  inline void read();
  inline void set(Z_NR<Z>& s);
  inline void set(Z& s);
  inline void set(unsigned long int s);
  inline void set_si(signed long int s);
  //  inline int sizeinbase2();
  inline double get_d_2exp(signed long int* expo);
  // virtual void tdiv_r_2exp(b,c)=0;
  inline Z& GetData();

  inline long int get_si();
  inline int sgn();

  inline int cmp(Z_NR<Z>& m);
  
  template<class T> inline void mul(Z_NR<Z>& a,T& b);
  
  inline void mul_si(Z_NR<Z>& a, signed long int b);
  inline void mul_si(Z_NR<Z>& a,Z_NR<long int>& b);
  inline void mul_ui(Z_NR<Z>& a, unsigned long int b);
  inline void add(Z_NR<Z>& a,Z_NR<Z>& b);
  inline void mul_2exp(Z_NR<Z>& a,long int expo);
  inline void div_2exp(Z_NR<Z>& a,long int expo);
  inline void addmul_ui(Z_NR<Z>& a,unsigned long int b);
  inline void submul_ui(Z_NR<Z>& a,unsigned long int b);
  

  template<class T> inline void addmul(Z_NR<Z>& a,T& b);
  template<class T> inline void submul(Z_NR<Z>& a,T& b);
 
  inline void add_ui(Z_NR<Z>& a,unsigned int b);
  inline void sub(Z_NR<Z>& a,Z_NR<Z>& b);
  inline void abs(Z_NR<Z>& a);
  inline void randb(int bits);
  inline void randm(Z_NR<Z>& max);
};

template<class F>
class FP_NR
{
  F data;
 public:
  typedef Z_NR<long int> assoc_int;
  inline FP_NR<F>();
  inline ~FP_NR<F>();
  inline void print();
  inline void printerr();
  inline void set(FP_NR<F>& s);
  inline void set(double s);
  inline void set(unsigned int s);
  inline double get();
  inline int exp();
  inline void neg(FP_NR<F>& b);
  inline void mul(FP_NR<F>& b,FP_NR<F>& c);
  inline void add(FP_NR<F>& b,FP_NR<F>& c);
  inline void sub(FP_NR<F>& b,FP_NR<F>& c);
  inline void div(FP_NR<F>& b,FP_NR<F>& c);
  inline void div_2ui(FP_NR<F>& b,unsigned int c);
  inline void mul_2ui(FP_NR<F>& b,unsigned int c);
  inline void abs(FP_NR<F>& b);
  inline int cmp(FP_NR<F>& b);
  inline int cmp(double d);
  inline int sgn();
  inline void rnd(FP_NR<F>& b);
  inline int zero_p();
  inline void set_nan();
  inline int is_nan();
  inline void sqrt(FP_NR<F>& b);
  inline signed long int get_si();/*long = au moins 32 bits, int = au moins 16 bits*/
  inline F& GetData();
  inline void submul(FP_NR<F>& b,FP_NR<F>& c);
  inline void addmul(FP_NR<F>& b, FP_NR<F>& c);
  static inline void setprec(unsigned int prec){};
};

/* specialization of FP_NR class for mpfr required to change type assoc_int */
template<> class FP_NR<mpfr_t>
{
  mpfr_t data;
 public:
  typedef Z_NR<mpz_t> assoc_int;
  inline FP_NR<mpfr_t>();
  inline ~FP_NR<mpfr_t>();
  inline void print();
  inline void printerr();
  inline void set(FP_NR<mpfr_t>& s);
  inline void set(double s);
  inline void set(unsigned int s);
  inline double get();
  inline int exp();
  inline void neg(FP_NR<mpfr_t>& b);
  inline void mul(FP_NR<mpfr_t>& b,FP_NR<mpfr_t>& c);
  inline void add(FP_NR<mpfr_t>& b,FP_NR<mpfr_t>& c);
  inline void sub(FP_NR<mpfr_t>& b,FP_NR<mpfr_t>& c);
  inline void div(FP_NR<mpfr_t>& b,FP_NR<mpfr_t>& c);
  inline void div_2ui(FP_NR<mpfr_t>& b,unsigned int c);
  inline void mul_2ui(FP_NR<mpfr_t>& b,unsigned int c);
  inline void abs(FP_NR<mpfr_t>& b);
  inline int cmp(FP_NR<mpfr_t>& b);
  inline int cmp(double d);
  inline int sgn();
  inline void rnd(FP_NR<mpfr_t>& b);
  inline int zero_p();
  inline void set_nan();
  inline int is_nan();
  inline void sqrt(FP_NR<mpfr_t>& b);
  inline signed long int get_si();/*long = au moins 32 bits, int = au moins 16 bits*/
  inline mpfr_t& GetData();
  inline void submul(FP_NR<mpfr_t>& b,FP_NR<mpfr_t>& c);
  inline void addmul(FP_NR<mpfr_t>& b, FP_NR<mpfr_t>& c);
  static inline void setprec(unsigned int prec){mpfr_set_default_prec (prec);};
};

#include "nr.cpp"

#endif
