/*
  Copyright 2005, 2006, 2007 David Cad, Damien Stehl.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehl, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/


#ifndef FAST_CPP
#define FAST_CPP

#include "fast.h"
#include "util.h"


/***********************************/
/* ******************************* */
/* Babai's Nearest Plane algorithm */
/* ******************************* */
/***********************************/

/* Size-reduces b_kappa using mu_ij and r_ij for j<=i <kappa
updates B (kappa)
computes mu_kappaj, r_kappaj for j<=kappa, and s(kappa) 
The algorithm is the iterative Babai algorithm of the paper
*/

template<class ZT,class FT>
inline int fast<ZT,FT>::BabaiCall(int*alpha,int zeros,int kappamax,int var_k,Z_NR<ZT>&ztmp,int& newvec,int& newvecmax,int n)
{
  return Babai (alpha[kappa], zeros, kappamax, var_k<=n?var_k:n, ztmp,kappa);
}


template<class ZT,class FT>
inline double
fast<ZT,FT>::GSO(int a, int zeros, int kappamax, int n,Z_NR<ZT>& ztmp,int aa,int rel)
{
  int j,k;
  double max=0.0;
  FT rtmp,tmp;
  for (j=aa; j<kappa; j++)
    {	  
      if (appSP[kappa][j]!=appSP[kappa][j])
	{
	  appSP[kappa][j] = fpScalarProduct (appB[kappa], appB[j], n);
	}
      
#ifdef DEBUG
	{
	  printf("\n          j is %d\n", j);	  
#ifdef FORCE_LONGDOUBLE
	  printf("          appSP[%d][%d] is: %1.5Le", 
		 kappa, j, appSP[kappa][j]); 
#else
	  printf("          appSP[%d][%d] is: %E", 
		 kappa, j, appSP[kappa][j]); 
#endif
	  printf(", which approximates ");
	  ScalarProduct (ztmp, B->GetVec(kappa), B->GetVec(j), n);
	  printf("\n");
	  B->print(kappa+1, n);
	  ztmp.print(); 
	  printf("\n          Norm of B[%d]^2: ", j);
	  ScalarProduct (ztmp, B->GetVec(j), B->GetVec(j), n);
	  ztmp.print();  
	  printf("\n          Norm of B[%d]^2: ",kappa);
	  ScalarProduct (ztmp, B->GetVec(kappa), B->GetVec(kappa), n);
	  ztmp.print(); 
	}
#endif
	  
      if (j > zeros+2)
	{
	  tmp = mu[j][zeros+1] * r[kappa][zeros+1];
	  rtmp = appSP[kappa][j] - tmp;
	  for (k=zeros+2; k<j-1; k++)
	    {
	      tmp = mu[j][k] * r[kappa][k];
	      rtmp = rtmp - tmp;
	    }
	  tmp = mu[j][j-1] * r[kappa][j-1];
	  r[kappa][j] = rtmp - tmp;
	}
      else if (j==zeros+2)
	{
	  tmp = mu[j][zeros+1] * r[kappa][zeros+1];
	  r[kappa][j] = appSP[kappa][j] - tmp;
	}
      else r[kappa][j] = appSP[kappa][j];
      

      mu[kappa][j] = r[kappa][j] / r[j][j];
      int x=expo[kappa]-expo[j];
      if (max<x) max=x;

    }
  return max;
}

template<class ZT,class FT> int 
fast<ZT,FT>::Babai (int a, int zeros, int kappamax, int n, Z_NR<ZT>& ztmp,int rel)
{
  int i, j, k, test, aa, exponent;
  signed long xx;
  FT tmp, rtmp;
#ifdef DEBUG
  int loops=0;
#endif 
  double max=0.0, max2=0.0, max3=0.0;

  aa = (a > zeros) ? a : zeros+1;
  
#ifdef DEBUG
  printf("\nappSP: \n");
  Print_matf (appSP, expo, kappamax+1, kappamax+1);
  printf("\nr: \n");
  Print_matf (r, expo, kappamax+1, kappamax+1);
  printf("\n          STARTING BABAI WITH k=%d\n", kappa);
  printf("\nappB: \n");
  Print_matf (appB, expo, kappamax+1, n);
  loops = 0;
  printf("\nmu: \n");
  Print_matf (mu, expo, kappamax+1, kappamax+1);
  printf ("\n\na is %d, zeros is %d, aa is %d\n", a, zeros, aa);
  B->print(kappamax+1, n);
#endif

  int ll=0;
  do
    {
      ll++;
      test=0;
      
#ifdef DEBUG     
      if (loops++ > LOOPS_BABAI) 
	{
	  printf("INFINITE LOOP?\n"); 
	  abort();
	}
#endif 
      
      /* ************************************** */
      /* Step2: compute the GSO for stage kappa */
      /* ************************************** */
      max3=max2;
      max2=max;
      max=GSO(a,zeros,kappamax,n,ztmp,aa,0);

      if(ll>=3)
	{
	  #ifdef DEBUG
	  cout << "\nrtmp="<<rtmp<<"\nmax2="<<max2<<"\nmax3="<<max3;
	  cout<<"\n";
	  #endif
	  if( max3!=max3 || max3<max2+SIZE_RED_FAILURE_THRESH)
	    {
#ifdef VERBOSE
	      cerr << "unexpected behaviour -> exit";
#endif
	      return kappa;
	    }
	}

#ifdef DEBUG
      if (loops <=LOOPS_BABAI)
	{
	  printf("\nmu :\n");
	  Print_matf (mu, expo, kappa+1, kappa+1);
	  printf("\nr :\n");
	  Print_matf (r, expo, kappa+1, kappa+1);
	}
#endif
      
      /* **************************** */
      /* Step3--5: compute the X_j's  */
      /* **************************** */
      
      for (j=kappa-1; j>zeros; j--)
	{
	  /* test of the relaxed size-reduction condition */

	  tmp = fabs (mu[kappa][j]);
	  tmp = ldexp (tmp, expo[kappa]-expo[j]);


#ifdef DEBUG
#ifdef FORCE_LONGDOUBLE
	  if (loops<=LOOPS_BABAI) printf( "tmp is: %1.5Le\n", tmp); 
#else
	  if (loops<=LOOPS_BABAI) printf( "tmp is: %E\n", tmp); 
#endif
#endif
	  
	  if (tmp > halfplus) 
	    {
	      test = 1; 
	      exponent = expo[j] - expo[kappa];
	      
	      /* we consider separately the cases X = +-1 */     
	      if (tmp <= onedothalfplus)   
		{
#ifdef DEBUG
		  printf(" X is pm1\n");
#endif
		  
		  if ( mu[kappa][j] >=0 )   /* in this case, X is 1 */
		    {
		      for (k=zeros+1; k<j; k++)
			{
			  tmp = ldexp (mu[j][k], exponent);
			  mu[kappa][k] =  mu[kappa][k] - tmp; 
			}
		      
		      for (i=0; i<n; i++)
			B->Get(kappa,i).sub(
			      B->Get(kappa,i), 
			      B->Get(j,i));	  
		    }
		  
		  else          /* otherwise X is -1 */ 
		    {
		      for (k=zeros+1; k<j; k++)
			{
			  tmp = ldexp (mu[j][k], exponent);
			  mu[kappa][k] = mu[kappa][k] + tmp;
			}
		      
		      for (i=0; i<n; i++)
			B->Get(kappa,i).add(
			      B->Get(kappa,i), 
			      B->Get(j,i));
		    }
		}
	      
	      else   /* we must have |X| >= 2 */
		{
		  tmp = ldexp (mu[kappa][j] , -exponent);

	          if ((tmp < static_cast<FT>( MAX_LONG_FAST))
		      &&(tmp > static_cast<FT>( -MAX_LONG_FAST)))  
		    {
		      tmp = rint (tmp);
		      
		      for (k=zeros+1; k<j; k++)
			{
			  rtmp = tmp * mu[j][k];
			  rtmp = ldexp (rtmp, exponent);
			  mu[kappa][k] = mu[kappa][k] - rtmp;
			}

		      xx = static_cast<signed long int>( tmp);
		      
#ifdef DEBUG
		      if (loops<=LOOPS_BABAI)
			{
			  printf("          xx[%d] is %ld\n", j, xx);
#ifdef FORCE_LONGDOUBLE
			  printf("          and tmp was %1.5Le\n", tmp);
#else
			  printf("          and tmp was %E\n", tmp);
#endif
			}
#endif

		      for (i=0; i<n; i++)
			{
			  if (xx > 0)
			    {
			      B->Get(kappa,i).submul_ui( 
					  B->Get(j,i), 
					  (unsigned long int) xx);
			    }
			  else
			    {
			      B->Get(kappa,i).addmul_ui( 
					  B->Get(j,i), 
					  (unsigned long int) -xx);
			    }
			}
		    }
		  
		  else
		    {
		      tmp = frexp(mu[kappa][j], &exponent); 
		      tmp = tmp * MAX_LONG_FAST;
		      xx = (signed long int) tmp;
		      exponent += expo[kappa]-expo[j] - CPU_SIZE_1 ;

#ifdef DEBUG
#ifdef FORCE_LONGDOUBLE
		      printf("tmp is %1.5Le", tmp);
#else
		      printf("tmp is %E", tmp);
#endif
		      printf("\nand exponent is %d, and X is %ld\n", 
			     exponent, xx);
#endif

		      /* This case is extremely rare: never occured to me */
		      if (exponent <= 0) 
			{
#ifdef VERBOSE
			  fprintf(stderr, 
				  "This is the rare Babai case\n");
#endif
			  xx = xx << -exponent;
			  exponent = 0;
			  
			  for (i=0; i<n; i++)
			    {
			      ztmp.mul_si(B->Get(j,i), xx);
			      B->Get(kappa,i).sub( 
				    B->Get(kappa,i), ztmp);
			    }
			  for (k=zeros+1; k<j; k++)
			    {
			      rtmp = ((FT) xx) * mu[j][k];
			      rtmp = ldexp (rtmp, expo[j]-expo[kappa]);
			      mu[kappa][k] = mu[kappa][k] - rtmp;
			    }
			  
			}
		      else
			{
			  for (i=0; i<n; i++)
			    {
			      ztmp.mul_2exp(B->Get(j,i), exponent); 
			      if (xx>0)
				{
				  B->Get(kappa,i).submul_ui( 
					      ztmp, 
					      (unsigned long int) xx);
				}
			      else
				{
				  B->Get(kappa,i).addmul_ui( 
					      ztmp, 
					      (unsigned long int) -xx);
				}
			    }
			  
			  for (k=zeros+1; k<j; k++)
			    {
			      rtmp = ((FT) xx) * mu[j][k];
			      rtmp = ldexp (rtmp, 
					    exponent+expo[j]-expo[kappa]);
			      mu[kappa][k] = mu[kappa][k] - rtmp;
			    }
			}
		    }		  
		}
	    }
	}
      
      if (test)   /* Anything happened? */
	{
	  expo[kappa] = set_line (appB[kappa], B->GetVec(kappa), n);
	  aa = zeros+1;
	  for (i=zeros+1; i<=kappa; i++) 
	    appSP[kappa][i] = NAN;//0.0/0.0;
	  for (i=kappa+1; i<=kappamax; i++) 
	    appSP[i][kappa] = NAN;//0.0/0.0;
	}
	  
#ifdef DEBUG
      if (loops<=LOOPS_BABAI)
	{
	  printf("          test is %d\n", test);
	  printf("\nmu: \n");
	  Print_matf (mu, expo, kappa+1, kappa+1);
	  printf("\nr: \n");
	  Print_matf (r, expo, kappa+1, kappa+1);
	}
#endif

    }
  while (test);



  if (appSP[kappa][kappa]!=appSP[kappa][kappa]) 
    {
      appSP[kappa][kappa] = fpNorm (appB[kappa], n);
    }
  s[zeros+1] = appSP[kappa][kappa];
  
  /* the last s[kappa-1]=r[kappa][kappa] is computed only if kappa increases */
  for (k=zeros+1; k<kappa-1; k++)
    {
      tmp = mu[kappa][k] * r[kappa][k];
      s[k+1] = s[k] - tmp;
    }

#ifdef DEBUG
  printf("          Number of loops is %d\n", loops);
#endif
  return 0;
}




/* ****************** */
/* ****************** */
/* ****************** */
/* The LLL Algorithm  */
/* ****************** */
/* ****************** */
/* ****************** */


/* LLL-reduces the integer matrix B "in place" */

template<class ZT,class FT>
int
fast<ZT,FT>::LLL ()
{
  int  kappa2, d, n, i, j, zeros, kappamax;
  FT  *mutmp, *appBtmp, *appSPtmp;
  FT tmp=0.0;
  
  int newvec=0;
  int newvecmax=1;
  int  *alpha;
  Z_NR<ZT> ztmp;
  Z_NR<ZT> *Btmp;
#ifdef VERBOSE
  int newkappa, start, lovasz, loops;
#endif
  
  n = B->GetNumCols();
  d = B->GetNumRows();

  unsigned long long int iter=0;
  unsigned long long int maxIter=(unsigned long long)(d - 2*d*(d+1)*((B->getMaxExp()+3)/log (delta)));

 

#ifdef DEBUG
  cerr<<"d = "<<d<<", n="<<n<<"\n";
#endif

#ifdef VERBOSE
#ifdef FORCE_LONGDOUBLE
  FT y;
  tmp = 1.0;
  for (j = 0; (y = tmp + 1.0) != tmp; j++, tmp = 2.0 * tmp);

  fprintf (stderr, 
	   "\nEntering fpLLL:\nLLL-reduction factors (%1.5Le, %1.5Le)\n", 
	   ctt, halfplus);  
  fprintf (stderr, 
	   "Working precision set to %d bits.\n", j);
#else
  fprintf (stderr, 
	   "\nEntering fpLLL:\nLLL-reduction factors (%E, %E)\n", 
	   ctt, halfplus);

  fprintf (stderr, 
	   "Working precision set to 53 bits.\n");
#endif
  fprintf (stderr, 
	   "No efficiency nor correctness guarantee here.\nIf you want to be sure, use the guaranteed version.\n\n");
#endif

  alpha = new int[d];
  expo = new int[d];


  mu = new FT*[d];
  for (i=0; i<d; i++)
    mu[i] = new FT[d];

  r = new FT*[d];
  for (i=0; i<d; i++)
    r[i] = new FT[d];

  appB = new FT*[d];
  for (i=0; i<d; i++)
    appB[i] = new FT[n];

  appSP = new FT*[d];
  for (i=0; i<d; i++)
    appSP[i] = new FT[d];

  s = new FT[d];
  appSPtmp = new FT[d];

  for (i=0; i<d; i++)
    for (j=0; j<d; j++)
      appSP[i][j] = NAN;
  

  
  /* ************************** */
  /* Step1: Initialization Step */
  /* ************************** */     
  
#ifdef VERBOSE  
  start = cputime();
#endif
  
  for (i=0; i<d; i++){
    expo[i] = set_line (appB[i], B->GetVec(i), n);
  }

#ifdef DEBUG  
  Print_matf (appB, expo, d, n);
  printf("Step 1 is over\n");
#endif  
  
  
  /* ********************************* */
  /* Step2: Initializing the main loop */
  /* ********************************* */   
  
  
#ifdef VERBOSE  
  newkappa = 0;
  loops = lovasz = 0;
#endif

  kappamax = 0;
  i = 0; 
  
  do
    appSP[i][i] = fpNorm (appB[i], n); 
  while ( (appSP[i][i] <=0.0)&&(++i <d));
  zeros = i-1; /* all vectors B[i] with i <= zeros are zero vectors */
  kappa = i+1;
  
  if (zeros < d-1)  r[i][i] = appSP[i][i];

  for (i=zeros+1; i<d; i++)
    alpha[i]=0;
  
#ifdef DEBUG
  Print_matf (appSP, expo, d, d);
  printf("Step 2 is over\n"); 
#endif  
  
  while (kappa < d)
    {      
      if (kappa>kappamax) 
	{
	  kappamax++;
	  newvec++;
	}
#ifdef VERBOSE
      loops++;
      if (kappa>newkappa)
	{
	  newkappa++;
	  fprintf (stderr, 
		   "Discovering vector k = %d, iterations = %d ", 
		   kappa+1, loops);
	  fprintf(stderr, "cputime is %d ms\n", (cputime()-start));
	}
#endif
      
#ifdef DEBUG
      printf("alpha= ");
      for (i=zeros+1; i<d; i++) 
	printf("%d ", alpha[i]);
      printf("\n");
      printf("entering the while loop with k=%d\n", kappa);
      B->print( d, n);
#endif

      /* ********************************** */
      /* Step3: Call to the Babai algorithm */
      /* ********************************** */   
      int var_k=kappamax+1+shift;
      if (BabaiCall(alpha,zeros,kappamax,var_k,ztmp,newvec,newvecmax,n))
	{

	  delete[] alpha;
	  delete[] expo;
	  
	  
	  clear_matrixf (mu, d);
	  clear_matrixf (r, d);
	  clear_matrixf (appB, d);
	  clear_matrixf (appSP, d);
	  delete[] s;
	  delete[] appSPtmp;
	  
	  return kappa;
	}
	

#ifdef DEBUG      
      printf("Step 3 is over\n");   
      B->print(kappamax+1, n);
      Print_matf (appSP, expo, kappamax+1, kappamax+1);
      printf("\nr: \n");
      Print_matf (r, expo, kappamax+1, kappamax+1);
#endif
      
      /* ************************************ */
      /* Step4: Success of Lovasz's condition */
      /* ************************************ */  
      /* ctt * r.coeff[kappa-1][kappa-1] <= s[kappa-2] ?? */
      

      tmp = r[kappa-1][kappa-1] * ctt;
      tmp = ldexp (tmp, 2*(expo[kappa-1]-expo[kappa]));


#ifdef DEBUG
#ifdef FORCE_LONGDOUBLE
      printf("s[%d] is %1.5Le\n", kappa-1, s[kappa-1]);
#else
      printf("s[%d] is %E\n", kappa-1, s[kappa-1]);
#endif
#endif
      
#ifdef VERBOSE
      lovasz++;
#endif

      if (tmp <= s[kappa-1]) 
	{
	  alpha[kappa] = kappa;
	  tmp = mu[kappa][kappa-1] * r[kappa][kappa-1];
	  r[kappa][kappa] = s[kappa-1] - tmp;
	  kappa++;
	} 
      else
	{

	  /* ******************************************* */
	  /* Step5: Find the right insertion index kappa */
          /* kappa2 remains the initial kappa            */
	  /* ******************************************* */  


	  kappa2 = kappa;
	  do
	    {
#ifdef VERBOSE
	      lovasz++;
#endif
	      kappa--;
	      if (kappa>zeros+1) 
		{
		  tmp = r[kappa-1][kappa-1] * ctt;
		  tmp = ldexp (tmp, 2*(expo[kappa-1]-expo[kappa2]));
		}
	    }
	  while ( (kappa >= zeros+2) && (s[kappa-1] <= tmp) );

#ifdef DEBUG
	  printf( "Index of insertion: %d \n", kappa);
	  printf("Step 5 is over\n");
	  printf("alpha= ");
	  for (i=0; i<=kappamax; i++) 
	    printf("%d ", alpha[i]);
	  printf("\n");
#endif
	  
	  for (i=kappa; i<kappa2; i++)
	    if (kappa <= alpha[i]) alpha[i] = kappa;

	  for (i=kappa2; i>kappa; i--)
	    alpha[i] = alpha[i-1];

	  for (i=kappa2+1; i<=kappamax; i++)
	    if (kappa < alpha[i]) alpha[i] = kappa;
	  
	  alpha[kappa] = kappa;

#ifdef DEBUG
	  printf("alpha= ");
	  for (i=0; i<=kappamax; i++) 
	    printf("%d ", alpha[i]);
	  printf("\n");
#endif

	  /* ****************************** */
	  /* Step6: Update the mu's and r's */
	  /* ****************************** */  
	  
	  mutmp = mu[kappa2];
	  for (i=kappa2; i>kappa; i--)
	    mu[i] = mu[i-1];
	  mu[kappa] = mutmp;
	  
	  mutmp = r[kappa2];
	  for (i=kappa2; i>kappa; i--)
	    r[i] = r[i-1];
	  r[kappa] = mutmp;

	  r[kappa][kappa] = s[kappa];
	  
#ifdef DEBUG 
	  printf("Step 6 is over\n");
#endif
	  
	  /* ************************ */
	  /* Step7: Update B and appB */
	  /* ************************ */  	  
	  
	  Btmp = B->GetVec(kappa2);
	  for (i=kappa2; i>kappa; i--)
	    B->GetVec(i) = B->GetVec(i-1);
	  B->GetVec(kappa) = Btmp;
	  
	  appBtmp = appB[kappa2];
	  for (i=kappa2; i>kappa; i--)
	    appB[i] = appB[i-1];
	  appB[kappa] = appBtmp;

	  j = expo[kappa2];
	  for (i=kappa2; i>kappa; i--)
	    expo[i] = expo[i-1];
	  expo[kappa] = j;

	  
#ifdef DEBUG
	  B->print(d, n);
	  Print_matf (appB, expo, d, n);
	  printf("Step 7 is over\n");
#endif

	  /* *************************** */
	  /* Step8: Update appSP: tricky */
	  /* *************************** */  	 
	  
	  for (i=0; i<=kappa2; i++)
	    appSPtmp[i] = appSP[kappa2][i];

	  for (i=kappa2+1; i<=kappamax; i++)
	    appSPtmp[i] = appSP[i][kappa2];
	  
	  for (i=kappa2; i>kappa; i--)
	    {
	      for (j=0; j<kappa; j++)
		appSP[i][j] = appSP[i-1][j];
	      
	      appSP[i][kappa] = appSPtmp[i-1];
	      
	      for (j=kappa+1; j<=i; j++)
		appSP[i][j] = appSP[i-1][j-1];

	      for (j=kappa2+1; j<=kappamax; j++)
		appSP[j][i] = appSP[j][i-1];     
	    }
	  
	  for (i=0; i<kappa; i++)
	    appSP[kappa][i] = appSPtmp[i];

	  appSP[kappa][kappa] = appSPtmp[kappa2];

	  for (i=kappa2+1; i<=kappamax; i++)
	    appSP[i][kappa] = appSPtmp[i];
	  
	  if (r[kappa][kappa] <= 0.0)
	    {
	      zeros++;
	      kappa++;
	      appSP[kappa][kappa] = fpNorm (appB[kappa], n);
	      r[kappa][kappa] = appSP[kappa][kappa];
	    }
	  
	  kappa++;
	  
#ifdef DEBUG	  
	  B->print( kappamax+1, n);
	  Print_matf (appSP, expo, kappamax+1, kappamax+1);
	  printf("Step 8 is over, new kappa=%d\n",kappa);
#endif

	}

      iter++;
      if (iter> maxIter)
	{
	  cerr<<"Too many loop iterations";
	  delete[] alpha;
	  delete[] expo;
	  
	  
	  clear_matrixf (mu, d);
	  clear_matrixf (r, d);
	  clear_matrixf (appB, d);
	  clear_matrixf (appSP, d);
	  delete[] s;
	  delete[] appSPtmp;

	  return -1;
	}
    };  
  

#ifdef VERBOSE
  tmp = 1.0;
  for (i = zeros+1; i<d; i++)
    tmp*=r[i][i];
  tmp = sqrt (tmp);

  kappa = 0;
  for (i=zeros+1; i<d; i++)
    kappa += expo[i];

  fprintf (stderr, "\nLoop iterations = %d \n", loops);
  fprintf (stderr, "Lovasz tests = %d \n", lovasz);
  fprintf (stderr, "Cputime is %d ms\n", (cputime()-start));
  if (zeros < d-1)
    {
#ifdef FORCE_LONGDOUBLE
      fprintf (stderr, "Vol(L) is %1.5Le * 2^%d\n", tmp, kappa);      
      fprintf (stderr, "||b_1|| is %1.5Le * 2^%d\n\n", 
	       sqrt ( r[zeros+1][zeros+1]), expo[zeros+1]);
#else
      fprintf (stderr, "Vol(L) is %E * 2^%d\n", tmp, kappa);      
      fprintf (stderr, "||b_1|| is %E * 2^%d\n\n", 
	       sqrt ( r[zeros+1][zeros+1]), expo[zeros+1]);
#endif
    }
#endif

  delete[] alpha;
  delete[] expo;

  
  clear_matrixf (mu, d);
  clear_matrixf (r, d);
  clear_matrixf (appB, d);
  clear_matrixf (appSP, d);
  delete[] s;
  delete[] appSPtmp;

  return 0;
}

template<class ZT,class FT>
fast<ZT,FT>::fast (ZZ_mat<ZT>* B_,int precision,double et,double delt)
{
  prec=precision;
  eta=et;
  delta=delt;

  halfplus=eta;
  onedothalfplus=1.0+eta;
  ctt=delta;

  B=B_;
  shift=B->getShift();
}


template<class ZT,class FT> inline ZZ_mat<ZT>* fast<ZT,FT>::GetBase()
{
  return B;
}

template<class ZT,class FT> fast<ZT,FT>::~fast()
{
}


#endif
