/***************************************************************************
                          FLFieldMetaData.h  -  description
                             -------------------
    begin                : Mon Jul 2 2001
    copyright            : (C) 2001,2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FLFIELDMETADATA_H
#define FLFIELDMETADATA_H

#include <qvariant.h>
#include <qstring.h>
#include <qlist.h>

class FLRelationMetaData;

/** Mantiene la descripcion/metadatos de una campo de una tabla.

	Esta clase solo tiene sentido cuando es parte de un objeto FLTableMetaData.

   	@author Federico Albujer Zornoza
   	@version 0.4*/
class FLFieldMetaData
{
	public:
	
	/** Constantes de tipos especiales no contemplados en QVariant */
	enum specialType
	{
		/** Tipo SERIAL o autoincremento */
		Serial = 100,
		Sequence = 200,
      Unlock = 300    /* Tipo de campo unlock: por Andrs Otn Urbano */
	};

	/** constructor.

      	@param n	Nombre del campo
      	@param a	Alias del campo, utilizado en etiquetas de los formularios
      	@param aN	TRUE si permite nulos (NULL), FALSE si los permite (NOT NULL)
      	@param iPK	TRUE si es clave primaria, FALSE si no es clave primaria, ser clave primaria implica ser Indice y nico
      	@param t	Tipo del campo
      	@param l	Longitud del campo en caracteres, siempre que se de tipo cadena de caracteres
      	@param c    Indica si el campo es calculado
      	@param v	Indica si el campo es visible
      	@param pI	Indica el nmero de dgitos de la parte entera
      	@param pD	Indica el nmero de decimales
      	@param iNX	TRUE si el campo es ndice
      	@param uNI	TRUE si el campo determina registros nicos
		@param coun	Indica si es un contador. Para referencias automticas.
		@param defValue Valor por defecto para el campo. */
	FLFieldMetaData (const QString & n, const QString & a, const bool aN,const bool iPK, const int t, const int l = 0,
						const bool c = false, const bool v=true, const bool ed=false, const int pI=4, const int pD=0, const bool iNX=false,
						const bool uNI=false, const bool coun=false, const QVariant defValue=QVariant()); 			

	/** desctructor */
   	~FLFieldMetaData ();

	/** Obtiene el nombre del campo.

      @return Nombre del campo */
  	const QString & name () const
  	{
    		return name_;
  	}

	/** Obtiene el alias del campo.

      	@return Alias del campo */
  	const QString & alias () const
  	{
    		return alias_;
  	}

	/** Obtiene si permite nulos.

      	@return TRUE si permite nulos, FALSE en caso contrario */
  	const bool allowNull () const
  	{
    		return allowNull_;
  	}

	/** Obtiene si es clave primaria.

      	@return TRUE si es clave primaria, FALSE en caso contrario */
  	const bool isPrimaryKey () const
  	{
    		return isPrimaryKey_;
  	}

	/** Obtiene el tipo del campo.

      	@return	El tipo del campo */
  	const int type () const
  	{
    		return type_;
  	}

	/** Obtiene la longitud del campo.

      	@return	La longitud del campo */
  	const int length () const
  	{
    		return length_;
  	}

  	/** Obtiene si el campo es calculado.

      	@return	TRUE si el campo es calculado, FALSE en caso contrario */
  	const bool calculated () const
  	{
    		return calculated_;
  	}
  	
  	/** Establece si el campo es calculado.
  	
  		@param c 	Valor TRUE si se quiere poner el campo como calculado, FALSE en caso contrario */
  	void setCalculated(const bool c)
  	{
		calculated_=c;  		
  	}

   /** Obtiene si el campo es editable.

    @return TRUE si el campo es editable, FALSE en caso contrario */
    const bool editable() const
    {
      return editable_;
    }

    /** Establece si el campo es editable.

    @param ed . Valor TRUE si se quiere que el campo sea editable, FALSE
      en caso contrario */
    void setEditable(const bool ed)
    {
      editable_=ed;
    }
  	/** Obtiene si el campo es visible.

      	@return	TRUE si el campo es visible, FALSE en caso contrario */
  	const bool visible () const
  	{
    		return visible_;
  	}
  	
  	/** Establece si el campo es visible.
  	
  		@param v 	Valor TRUE si se quiere poner el campo como visible, FALSE
  					en caso contrario */
  	void setVisible(const bool v)
  	{
		visible_=v;  		
  	}
  	
  	/** Obtiene el nmero de dgitos de la parte entera.

      	@return	El nmero de dgitos de la parte entera del campo */
  	const int partInteger () const
  	{
    		return partInteger_;
  	}
  	
  	/** Obtiene si el nmero de dgitos de la parte decimal.

      	@return	El nmero de dgitos de la parte decimal del campo */
  	const int partDecimal () const
  	{
    		return partDecimal_;
  	}

   /** Obtiene si el campo es contador.

       @return     TRUE si el campo es una referencia con contador.*/
    const bool isCounter () const
    {
        return contador_;
    }
  	
  	/** Obtiene si el campo es ndice.

      	@return	TRUE si el campo es ndice, FALSE en caso contrario */
  	const bool isIndex () const
  	{
    		return isIndex_;
  	}
  	
  	/** Obtiene si el campo determina registros nicos.

      	@return	TRUE si el campo determina registros nicos, FALSE en caso contrario */
  	const bool isUnique () const
  	{
    		return isUnique_;
  	}
  	
    /** Tipo de datos lista de relaciones */
  	typedef QList < FLRelationMetaData > FLRelationMetaDataList;

	/** Aade una relacion con otra tabla para este campo.

      	Aade un nuevo objeto FLRelationMetaData, a la lista
      	de relaciones para este campo.

      	Hay que notar que para un campo solo puede existir una
      	sola relacion del tipo M1 (muchos a uno), por lo que en
      	el caso de que se quieran aadir varias relaciones de
      	este tipo para el campo solo se tendr en cuenta la primera.
      	Relaciones del tipo 1M (uno a muchos) pueden existir todas
      	las que sean necesarias. Ver FLRelationMetaData::Cardinality.

      	@param r	Objeto FlRelationMetaData con la definicion de la
                			relacion a aadir */
  	void addRelationMD (FLRelationMetaData * r);

    /**  Asigna una lista de relaciones, a la lista de relaciones del campo.

         La lista de relaciones del tipo FLRelationMetaDataList, ya construida,
         es asignada como la lista de relaciones del campo, en el caso de que
         ya exista una lista de relaciones para el campo, esta es destruida y
         sobreescrita por la nueva. La lista pasada a este mtodo pasa a ser
         propiedad del campo, y l es el encargado de borrarla, por lo tanto no se
         debe borrar esta lista fuera de la clase. Si la lista que se pretende asignar
         es nula o vaca este mtodo no hace nada.

         @param rl  Lista de relaciones */
    void setRelationList (FLRelationMetaDataList *rl);

	/** Para obtener la lista de definiciones de las relaciones.

      	@return Objeto con la lista de deficiones de la relaciones del campo */
  	FLRelationMetaDataList *relationList () const
  	{
      	return relationList_;
  	}

	/** Para obtener la relacion muchos a uno para este campo.

      	@return Objeto	FLRelationMetaData con la descripcion de la relacion
              				muchos a uno para este campo */
  	FLRelationMetaData *relationM1 () const
  	{
      	return relationM1_;
  	}

  	/** Establece un campo asociado para este campo, y el nombre
  	     del campo de la tabla fornea que se debe utilizar para filtrar segn
  	     el valor del campo asociado.
  	
  		Ver FLFieldMetaData::associatedField_
  		Ver FLFieldMetaData::associatedFieldFilterTo_
  		
  		@param r 	Objeto FLFieldMetaData que define el campo que se quiere asociar a este
  		@param f 	Nombre del campo a aplicar el filtro*/
  	void setAssociatedField(FLFieldMetaData * r, const QString & f)
  	{
  		associatedField_=r;
  		associatedFieldFilterTo_=f;
  	}
  	
  	/** Devuelve el campo asociado para este campo.
  	
  		Ver FLFieldMetaData::associatedField_
  		
  		@return Objeto FLFieldMetaData que define el campo asociado a este, o 0
  				si no hay campo asociado. */
  	FLFieldMetaData * associatedField() const
  	{
  		return associatedField_;
  	}
  	
	/** Devuelve el nombre del campo que hay que filtrar segn el campo asociado.
  	
  		Ver FLFieldMetaData::associatedFieldFilterTo_
  		
  		@return Nombre del campo de la tabla fornea M-1, al que hay que aplicar el filtro
  				segn el valor del campo asociado. */  	
  	const QString & associatedFieldFilterTo () const
  	{
    	return associatedFieldFilterTo_;
  	}

   /** Devuelve el valor por defecto para el campo.

        @return Valor que se asigna por defecto al campo. */
   QVariant defaultValue()
   {
      return defaultValue_;
   }
   
	private:

	/** Nombre del campo en la tabla */
   QString name_;

	/** Alias o mote para el campo, usado como
      	etiqueta de campos en los formularios */
  	QString alias_;

	/** Almacena si el campo permite ser nulo */
  	bool allowNull_;

	/** Almacena si el campo es clave primaria */
  	bool isPrimaryKey_;

	/** Tipo del campo */
  	int type_;

	/** Longitud del campo */
  	int length_;

  	/** Indica si el campo es calculado */
  	bool calculated_;
  	
  	/** Indica si el campo es visible */
  	bool visible_;

    /** Indica si el campo es editable */
    bool editable_;
  	
  	/** Indica el nmero de dgitos de la parte entera */
  	int partInteger_;
  	
  	/** Indica el nmeor de dgitos de la parte decimal */
  	int partDecimal_;
  	
  	/** Indica si el campo es ndice */
  	bool isIndex_;
  	
  	/** Indica si el campo es nico */
  	bool isUnique_;

   /** Indica si el campo es un contador de referencia y facturalux en el momento de insertar
        un registro debe intentar calcular cual sera el siguiente numero
        @author Andrs Otn Urbano (andresoton@eresmas.com)*/
   bool contador_;
  	
	/** Lista de relaciones para este campo */
  	FLRelationMetaDataList *relationList_;

	/** Mantiene, si procede, la relacin M1 (muchos a uno)
      	para el campo (solo puede haber una relacion de este tipo
      	para un campo) */
  	FLRelationMetaData *relationM1_;
  	
  	/** Asocia este campo con otro, para efectuar filtros en bsquedas.
  	
		El campo que se asocia a este debe tener una relacin M-1.
		Este campo tambin debe tener una relacin M-1. Al asociar un campo a este,
		las bsquedas mediante los botones de bsqueda en los formularios de edicin
		de registros vendrn condicionadas por el valor del campo asociado en el
		momento de realizar dicha bsqueda. Cuando se realiza una bsqueda para
		este campo la tabla relacionada con l (M-1) ser mostrada para elegir un
		registro de todos los posibles, en el caso normal se muestran todos los registros,
		pero cuando se asocia un campo slo se muestran aquellos registros que satisfagan el
		valor del campo asociado. Ejemplo : En la tabla albaranes asociamos el campo
		'codemporig' al campo 'codalmorig' (NO es lo mismo que asociar 'codalmorig'
		a 'codemporig') cuando abrimos el formulario de albaranes elegimos una empresa
		origen (codemporig), cuando vayamos a elegir un almacen origen (codalmorig) slo
		se podr elegir entre los almacenes que son de la empresa origen , ya que el formulario
		de bsqueda slo se mostrarn los almacenes cuyo cdigo de empresa (ver FLFieldMetaData::associatedFieldFilterTo_)
		sea igual al valor de la empresa origen elegida (codemporig). */
	FLFieldMetaData *associatedField_;
	
	/** Nombre del campo que se debe filtra segn el campo asociado.
		
		Esta propiedad slo tiene sentido cuando hay un campo asociado a este,
		ver FLFieldMetaData	::associatedField_ , y si ese campo tiene una relacion M-1. Indica
		el nombre del campo de la tabla fornea en la relacin M-1, que se debe utilizar para filtrar
		los registros segn el valor del campo asociado. Ejemplo : En la tabla albaranes asociamos el campo
		'codemporig' al campo 'codalmorig' (NO es lo mismo que asociar 'codalmorig'
		a 'codemporig'), e indicamos que el campo de filtro es 'codempresa' de la tabla relacionada M-1 con el
		campo 'codalmorig' (Almacenes) . Cuando abrimos el formulario de albaranes elegimos una empresa 		
		origen (codemporig), cuando vayamos a elegir un almacen origen (codalmorig) slo se podr elegir
		entre los almacenes que son de la empresa origen, ya que el formulario de bsqueda slo se mostrarn
		los almacenes cuyo cdigo de empresa (el campo indicado de filtro ) sea igual al valor de la empresa
		origen elegida (codemporig). */
	QString associatedFieldFilterTo_;

    /** Valor por defecto para el campo */
    QVariant defaultValue_;
};

#endif
