/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-storage-edit.c,v 1.16 2005/02/01 02:32:00 hoa Exp $
 */

#include "etpan-storage-edit.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-storage.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "etpan-cfg-edit-common.h"
#include "etpan-help-viewer.h"
#include "etpan-search-input.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_storage_editor_app_driver = {
  .name = "storage-edit",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .get_idle_udelay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

#define INFO_TYPE_MAX 20

enum {
  INFO_TYPE_NAME,
  INFO_TYPE_STORAGE_TYPE,
  INFO_TYPE_CONNECTION_TYPE,
  INFO_TYPE_HOSTNAME,
  INFO_TYPE_PORT,
  INFO_TYPE_COMMAND,
  INFO_TYPE_AUTH_TYPE,
  INFO_TYPE_LOGIN,
  INFO_TYPE_PASSWORD,
  INFO_TYPE_LOCATION,
};

enum {
  INFO_PRESENT,
  INFO_DEPEND_ON_COMMAND,
  INFO_DEPEND_ON_HOST,
};

#define TABLE_SIZE(a) (sizeof(a) / sizeof(a[0]))

enum {
  STORAGE_IS_HOST,
  STORAGE_IS_COMMAND,
};

struct app_state {
  int storage_type;
  
  char name[256];
  
  /* remote storage */
  unsigned int connection_type_index;
  unsigned int connection_type;
  char hostname[256];
  int port;
  char command[PATH_MAX];
  unsigned int auth_type_index;
  int auth_type;
  char login[256];
  char password[256];
  
  /* local storage */
  char location[PATH_MAX];
  
  unsigned int connection_type_count;
  unsigned int auth_type_count;
  unsigned int index;
  unsigned int first;
  unsigned int count;
  char info_type_tab[INFO_TYPE_MAX];
  
  int main_attr;
  int selection_attr;
  int status_attr;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

struct info_type {
  int type;
  int present;
};



static struct info_type local_storage_info_type[] = {
  { INFO_TYPE_NAME, INFO_PRESENT },
  { INFO_TYPE_STORAGE_TYPE, INFO_PRESENT },
  { INFO_TYPE_LOCATION, INFO_PRESENT },
};

static struct info_type remote_storage_info_type[] = {
  { INFO_TYPE_NAME, INFO_PRESENT },
  { INFO_TYPE_STORAGE_TYPE, INFO_PRESENT },
  { INFO_TYPE_CONNECTION_TYPE, INFO_PRESENT },
  { INFO_TYPE_HOSTNAME, INFO_DEPEND_ON_HOST },
  { INFO_TYPE_PORT, INFO_DEPEND_ON_HOST },
  { INFO_TYPE_COMMAND, INFO_DEPEND_ON_COMMAND },
  { INFO_TYPE_AUTH_TYPE, INFO_PRESENT },
  { INFO_TYPE_LOGIN, INFO_PRESENT },
  { INFO_TYPE_PASSWORD, INFO_PRESENT },
};

static struct etpan_type_name imap_auth_types_name[] = {
  {IMAP_AUTH_TYPE_PLAIN, "plain authentication"},
#if 0
  {IMAP_AUTH_TYPE_SASL_ANONYMOUS, "SASL anonymous"},
  {IMAP_AUTH_TYPE_SASL_CRAM_MD5, "SASL CRAM-MD5"},
  {IMAP_AUTH_TYPE_SASL_KERBEROS_V4, "SASL Kerberos V4"},
  {IMAP_AUTH_TYPE_SASL_PLAIN, "SASL plain authentication"},
  {IMAP_AUTH_TYPE_SASL_SCRAM_MD5, "SCRAM MD5"},
  {IMAP_AUTH_TYPE_SASL_GSSAPI, "GSSAPI"},
  {IMAP_AUTH_TYPE_SASL_DIGEST_MD5, "digest MD5"},
#endif
};

static struct etpan_type_name pop3_auth_types_name[] = {
  {POP3_AUTH_TYPE_PLAIN, "plain authentication"},
  {POP3_AUTH_TYPE_APOP, "APOP authentication ( login & password MD5 digest)"},
  {POP3_AUTH_TYPE_TRY_APOP, "try APOP authentication"},
#if 0
  {POP3_AUTH_TYPE_SASL_ANONYMOUS, "SASL anonymous"},
  {POP3_AUTH_TYPE_SASL_CRAM_MD5, "SASL CRAM-MD5"},
  {POP3_AUTH_TYPE_SASL_KERBEROS_V4, "SASL Kerberos V4"},
  {POP3_AUTH_TYPE_SASL_PLAIN, "SASL plain authentication"},
  {POP3_AUTH_TYPE_SASL_SCRAM_MD5, "SCRAM MD5"},
  {POP3_AUTH_TYPE_SASL_GSSAPI, "GSSAPI"},
  {POP3_AUTH_TYPE_SASL_DIGEST_MD5, "digest MD5"},
#endif
};

static struct etpan_type_name nntp_auth_types_name[] = {
  {NNTP_AUTH_TYPE_PLAIN, "plain authentication"},
};

static struct {
  int type;
  char * name;
  char * description;
  struct info_type * info_type_tab;
  unsigned int info_type_count;
  struct etpan_type_name * auth_type_tab;
  unsigned int auth_type_count;
} storage_type_tab[] = {
  {
    .type = STORAGE_TYPE_MBOX,
    .name = "mbox",
    .description = "mbox mailbox file",
    .info_type_tab = local_storage_info_type,
    .info_type_count = TABLE_SIZE(local_storage_info_type),
    .auth_type_tab = NULL,
    .auth_type_count = 0,
  },
  {
    .type = STORAGE_TYPE_MAILDIR,
    .name = "maildir",
    .description = "maildir mailbox directory",
    .info_type_tab = local_storage_info_type,
    .info_type_count = TABLE_SIZE(local_storage_info_type),
    .auth_type_tab = NULL,
    .auth_type_count = 0,
  },
  {
    .type = STORAGE_TYPE_DB,
    .name = "db",
    .description = "DB mailbox file",
    .info_type_tab = local_storage_info_type,
    .info_type_count = TABLE_SIZE(local_storage_info_type),
    .auth_type_tab = NULL,
    .auth_type_count = 0,
  },
  {
    .type = STORAGE_TYPE_MH,
    .name = "mh",
    .description = "MH mailbox directory",
    .info_type_tab = local_storage_info_type,
    .info_type_count = TABLE_SIZE(local_storage_info_type),
    .auth_type_tab = NULL,
    .auth_type_count = 0,
  },
  {
    .type = STORAGE_TYPE_POP3,
    .name = "pop3",
    .description = "POP3 remote mailbox",
    .info_type_tab = remote_storage_info_type,
    .info_type_count = TABLE_SIZE(remote_storage_info_type),
    .auth_type_tab = pop3_auth_types_name,
    .auth_type_count = TABLE_SIZE(pop3_auth_types_name),
  },
  {
    .type = STORAGE_TYPE_IMAP,
    .name = "imap",
    .description = "IMAP remote mailboxes",
    .info_type_tab = remote_storage_info_type,
    .info_type_count = TABLE_SIZE(remote_storage_info_type),
    .auth_type_tab = imap_auth_types_name,
    .auth_type_count = TABLE_SIZE(imap_auth_types_name),
  },
  {
    .type = STORAGE_TYPE_NNTP,
    .name = "nntp",
    .description = "NNTP newsgroups",
    .info_type_tab = remote_storage_info_type,
    .info_type_count = TABLE_SIZE(remote_storage_info_type),
    .auth_type_tab = nntp_auth_types_name,
    .auth_type_count = TABLE_SIZE(nntp_auth_types_name),
  },
};

static int get_storage_index(int value)
{
  unsigned int i;
  
  for(i = 0 ; i < TABLE_SIZE(storage_type_tab) ; i ++) {
    if (storage_type_tab[i].type == value)
      return i;
  }
  
  return -1;
}

static struct etpan_type_name connection_types_name[] = {
  {CONNECTION_TYPE_PLAIN, "clear-text connection"},
  {CONNECTION_TYPE_STARTTLS, "use STARTTLS on clear-text connection"},
  {CONNECTION_TYPE_TRY_STARTTLS,
   "try STARTTLS, fallback on clear-text connection"},
  {CONNECTION_TYPE_TLS, "TLS connection (SSL)"},
  {CONNECTION_TYPE_COMMAND, "using a command, clear-text connection"},
  {CONNECTION_TYPE_COMMAND_STARTTLS, "using a command, use STARTTLS on clear-text connection"},
  {CONNECTION_TYPE_COMMAND_TRY_STARTTLS,
   "using a command, try STARTTLS, fallback on clear-text connection"},
  {CONNECTION_TYPE_COMMAND_TLS, "using a command, TLS connection (SSL)"},
};

static int get_tab_index(struct etpan_type_name * tab,
    unsigned int count, int value)
{
  unsigned int i;
  
  for(i = 0 ; i < count ; i ++) {
    if (tab[i].id == value)
      return i;
  }
  
  return 0;
}

static int get_tab_index_default(struct etpan_type_name * tab,
    int count, int value, int default_index)
{
  int r;
  
  r = get_tab_index(tab, count, value);
  if (r == -1)
    return default_index;
  else
    return r;
}

static void edit_location(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void ask_quit(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  int current_type;
  unsigned int list_lines;

  state = app->data;
  
  list_lines = app->display_height - 1;
  
  current_type = state->info_type_tab[state->index];
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'h':
  case KEY_LEFT:
    switch (current_type) {
    case INFO_TYPE_STORAGE_TYPE:
      if (state->storage_type > 0)
        state->storage_type --;
      break;
    case INFO_TYPE_CONNECTION_TYPE:
      if (state->connection_type_index > 0)
        state->connection_type_index --;
      break;
    case INFO_TYPE_AUTH_TYPE:
      if (state->auth_type_index > 0)
        state->auth_type_index --;
      break;
    }

    break;
    
  case 'l':
  case KEY_RIGHT:
    switch (current_type) {
    case INFO_TYPE_STORAGE_TYPE:
      if (state->storage_type < (int) TABLE_SIZE(storage_type_tab) - 1)
        state->storage_type ++;
      break;
    case INFO_TYPE_CONNECTION_TYPE:
      if (state->connection_type_index < state->connection_type_count - 1)
        state->connection_type_index ++;
      break;
    case INFO_TYPE_AUTH_TYPE:
      if (state->auth_type_index < state->auth_type_count - 1)
        state->auth_type_index ++;
      break;
    }
    break;

  case 'k':
  case KEY_UP:
    if (state->index > 0)
      state->index --;
    break;
    
  case 'j':
  case KEY_DOWN:
    if (state->index < state->count - 1)
      state->index ++;
    break;
    
  case KEY_NPAGE:
    if (state->index + list_lines - 1 < state->count)
      state->index += list_lines - 1;
    else
      state->index = state->count - 1;
    break;
    
  case KEY_PPAGE:
    if (state->index >= list_lines - 1)
      state->index -= list_lines - 1;
    else
      state->index = 0;
    break;
    
  case KEY_HOME:
    state->index = 0;
    break;
    
  case KEY_END:
    state->index = state->count - 1;
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_EDIT_CANCEL, state->upcall_data);
    break;

  case 'y':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_EDIT_VALID, state->upcall_data);
    break;
    
  case '\n':
    switch (current_type) {
    case INFO_TYPE_NAME:
      etpan_cfg_edit_string(app, "Enter storage name: ",
          state->name, sizeof(state->name), 0);
      break;
    case INFO_TYPE_HOSTNAME:
      etpan_cfg_edit_string(app, "Enter hostname: ",
          state->hostname, sizeof(state->hostname), 0);
      break;
    case INFO_TYPE_PORT:
      etpan_cfg_edit_int(app, "Enter port: ", &state->port);
      break;
    case INFO_TYPE_COMMAND:
      etpan_cfg_edit_filename(app, "Enter command: ",
          state->command, sizeof(state->command));
      break;
    case INFO_TYPE_LOGIN:
      etpan_cfg_edit_string(app,  "Enter login: ",
          state->login, sizeof(state->login), 0);
      break;
    case INFO_TYPE_PASSWORD:
      etpan_cfg_edit_string(app,  "Enter password: ",
          state->password, sizeof(state->password),
          ETPAN_INPUT_COMMON_HIDDEN);
      break;
    case INFO_TYPE_LOCATION:
      edit_location(app);
      break;
    }
    break;
  }
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  struct info_type * info_type_tab;
  unsigned int info_type_count;
  struct etpan_type_name  * auth_type_tab;
  unsigned int auth_type_count;
  unsigned int count;
  unsigned int i;
  unsigned int y;
  unsigned int list_lines;
  char * buffer;
  char * output;
  char * fill;
  unsigned int percent;

  buffer = app->app->buffer;
  output = app->app->output;
  fill = app->app->fill;
  
  list_lines = app->display_height - 1;
  
  state = app->data;
  
  /* update view */
  
  info_type_tab = storage_type_tab[state->storage_type].info_type_tab;
  info_type_count = storage_type_tab[state->storage_type].info_type_count;
  
  /* connection type */
  
  state->connection_type =
    connection_types_name[state->connection_type_index].id;
  
  /* auth type */
  
  auth_type_tab = storage_type_tab[state->storage_type].auth_type_tab;
  auth_type_count = storage_type_tab[state->storage_type].auth_type_count;
  
  if (auth_type_tab != NULL) {
    state->auth_type_count = auth_type_count;
    if (state->auth_type_index > auth_type_count - 1)
      state->auth_type_index = auth_type_count - 1;
    
    state->auth_type = auth_type_tab[state->auth_type_index].id;
  }
  else {
    state->auth_type_index = 0;
    state->auth_type = 0;
  }
  
  count = 0;
  for(i = 0 ; i < info_type_count ; i ++) {
    int do_display;
    
    do_display = 0;
    switch(info_type_tab[i].present) {
    case INFO_PRESENT:
      do_display = 1;
      break;
      
    case INFO_DEPEND_ON_COMMAND:
      switch (state->connection_type) {
      case CONNECTION_TYPE_COMMAND:
      case CONNECTION_TYPE_COMMAND_STARTTLS:
      case CONNECTION_TYPE_COMMAND_TRY_STARTTLS:
      case CONNECTION_TYPE_COMMAND_TLS:
        do_display = 1;
        break;
      }
      break;
      
    case INFO_DEPEND_ON_HOST:
      switch (state->connection_type) {
      case CONNECTION_TYPE_PLAIN:
      case CONNECTION_TYPE_STARTTLS:
      case CONNECTION_TYPE_TRY_STARTTLS:
      case CONNECTION_TYPE_TLS:
        do_display = 1;
        break;
      }
      break;
    }
    
    if (do_display) {
      state->info_type_tab[count] = info_type_tab[i].type;
      count ++;
    }
  }
  
  state->count = count;

  if (state->index > state->count - 1)
    state->index = state->count - 1;
  
  if (state->index < state->first)
    state->first = state->index;
  if (state->index - state->first + 1 > (unsigned int) list_lines)
    state->first = state->index - list_lines + 1;
  
  /* display */
  
  /* duplicated but necessary if we want to separate update view and display */
  auth_type_tab = storage_type_tab[state->storage_type].auth_type_tab;
  
  wattron(w, state->main_attr);
  y = 0;
  i = state->first;
  while (y < list_lines) {
    const char * connection_type_str;
    const char * auth_type_str;
    
    if (i > count - 1)
      break;
    
    switch (state->info_type_tab[i]) {
    case INFO_TYPE_NAME:
      snprintf(buffer, app->display_width, "name: %s", state->name);
      break;

    case INFO_TYPE_STORAGE_TYPE:
      snprintf(buffer, app->display_width, "type: %s",
          storage_type_tab[state->storage_type].description);
      break;
      
    case INFO_TYPE_CONNECTION_TYPE:
      connection_type_str =
        connection_types_name[state->connection_type_index].name;
      snprintf(buffer, app->display_width, "%s", connection_type_str);
      break;
      
    case INFO_TYPE_HOSTNAME:
      snprintf(buffer, app->display_width, "hostname: %s",
          state->hostname);
      break;
      
    case INFO_TYPE_PORT:
      if (state->port == 0)
        snprintf(buffer, app->display_width, "port: (default port)");
      else
        snprintf(buffer, app->display_width, "port: %i",
            state->port);
      break;

    case INFO_TYPE_COMMAND:
      snprintf(buffer, app->display_width, "command: %s",
          state->command);
      break;
      
    case INFO_TYPE_LOGIN:
      snprintf(buffer, app->display_width, "login: %s",
          state->login);
      break;

    case INFO_TYPE_AUTH_TYPE:
      auth_type_str = auth_type_tab[state->auth_type_index].name;
      snprintf(buffer, app->display_width,
          "authentication: %s", auth_type_str);
      break;
      
    case INFO_TYPE_PASSWORD:
      snprintf(buffer, app->display_width, "password: (hidden)");
      break;
      
    case INFO_TYPE_LOCATION:
      snprintf(buffer, app->display_width, "path: %s",
          state->location);
      break;
    }
    if (i == state->index) {
      wattroff(w, state->main_attr);
      wattron(w, state->selection_attr);
    }
    snprintf(output, app->display_width, "%s%s", buffer, fill);
    mvwprintw(w, y, 0, "%s", output);
    if (i == state->index) {
      wattroff(w, state->selection_attr);
      wattron(w, state->main_attr);
    }
    
    i ++;
    y ++;
  }

  while (y < list_lines) {
    mvwprintw(w, y, 0, "%s", fill);
    y ++;
  }

  wattroff(w, state->main_attr);
  
  /* status bar */

  wattron(w, state->status_attr);
  
  if (state->count == 0)
    percent = 0;
  if (state->count == 1)
	percent = 100;
  else
    percent = state->index * 100 / (state->count-1);
  
  snprintf(output, app->display_width + 1,
      " %3i %% | y: ok  ^G: cancel%s", percent, fill);
  
  mvwprintw(w, app->display_height - 1, 0, "%s", output);
  
  wattroff(w, state->status_attr);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_app_set_color(app->app, "main",
      &state->main_attr, A_NORMAL);
  etpan_app_set_color(app->app, "selection",
      &state->selection_attr, A_REVERSE);
  etpan_app_set_color(app->app, "status",
      &state->status_attr, A_REVERSE);
}

static int init(struct etpan_subapp * app)
{
  struct app_state * state;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->storage_type = 0;

  state->name[0] = '\0';
  
  /* remote storage */
  state->hostname[0] = '\0';
  state->port = 0;
  state->command[0] = '\0';
  state->connection_type_index = 0;
  state->connection_type = 0;
  state->auth_type_index = 0;
  state->auth_type = 0;
  state->login[0] = '\0';
  state->password[0] = '\0';
  
  /* local storage */
  state->location[0] = '\0';;
  
  state->auth_type_count = 0;
  state->connection_type_count = TABLE_SIZE(connection_types_name);
  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  /* colors */
  state->main_attr = A_NORMAL;
  state->selection_attr = A_REVERSE;
  state->status_attr = A_REVERSE;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  app->data = state;
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  
  free(state);
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - edit storage");
  return etpan_app_subapp_display_init(app);
}

struct etpan_subapp * etpan_storage_editor_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_storage_editor_app_driver);
}

void etpan_storage_editor_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->storage_type = 0;

  state->name[0] = '\0';
  
  /* remote storage */
  state->hostname[0] = '\0';
  state->port = 0;
  state->command[0] = '\0';
  state->connection_type_index = 0;
  state->connection_type = 0;
  state->auth_type_index = 0;
  state->auth_type = 0;
  state->login[0] = '\0';
  state->password[0] = '\0';
  
  /* local storage */
  state->location[0] = '\0';;
  
  state->auth_type_count = 0;
  state->index = 0;
  state->first = 0;
  state->count = 0;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
}


static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_storage_editor_app_flush(app);
}

static void server_info_set(struct etpan_subapp * app,
    char * hostname, int port, char * command, int connection_type)
{
  struct app_state * state;
  
  state = app->data;
  
  if (hostname != NULL) {
    strncpy(state->hostname, hostname, sizeof(state->hostname));
    state->hostname[sizeof(state->hostname) - 1] = '\0';
    state->port = port;
  }
  
  if (command != NULL) {
    strncpy(state->command, command, sizeof(state->command));
    state->command[sizeof(state->command) - 1] = '\0';
  }
  
  state->connection_type_index =
    get_tab_index_default(connection_types_name,
        TABLE_SIZE(connection_types_name),
        connection_type, 0);
}

static void auth_info_set(struct etpan_subapp * app,
    char * login, char * password)
{
  struct app_state * state;
  
  state = app->data;

  if (login != NULL) {
    strncpy(state->login, login, sizeof(state->login));
    state->login[sizeof(state->login) - 1] = '\0';
  }
  if (password != NULL) {
    strncpy(state->password, password, sizeof(state->password));
    state->password[sizeof(state->password) - 1] = '\0';
  }
}

static void imap_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct imap_mailstorage * imap_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_IMAP);
  
  imap_data = storage->sto_data;
  
  server_info_set(app, imap_data->imap_servername, imap_data->imap_port,
      imap_data->imap_command, imap_data->imap_connection_type);
  
  state->auth_type_index = 
    get_tab_index_default(imap_auth_types_name,
        TABLE_SIZE(imap_auth_types_name),
        imap_data->imap_auth_type, 0);
  
  auth_info_set(app, imap_data->imap_login, imap_data->imap_password);
}

static void pop3_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct pop3_mailstorage * pop3_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_POP3);
  
  pop3_data = storage->sto_data;
  
  server_info_set(app, pop3_data->pop3_servername, pop3_data->pop3_port,
      pop3_data->pop3_command, pop3_data->pop3_connection_type);
  
  state->auth_type_index = 
    get_tab_index_default(pop3_auth_types_name,
        TABLE_SIZE(pop3_auth_types_name),
        pop3_data->pop3_auth_type, 0);
  
  auth_info_set(app, pop3_data->pop3_login, pop3_data->pop3_password);
}

static void nntp_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct nntp_mailstorage * nntp_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_NNTP);

  nntp_data = storage->sto_data;
  
  server_info_set(app, nntp_data->nntp_servername, nntp_data->nntp_port,
      nntp_data->nntp_command, nntp_data->nntp_connection_type);
  
  state->auth_type_index = 
    get_tab_index_default(nntp_auth_types_name,
        TABLE_SIZE(nntp_auth_types_name),
        nntp_data->nntp_auth_type, 0);
  
  auth_info_set(app, nntp_data->nntp_login, nntp_data->nntp_password);
}

static void mbox_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct mbox_mailstorage * mbox_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_MBOX);

  mbox_data = storage->sto_data;
  
  strncpy(state->location, mbox_data->mbox_pathname, sizeof(state->location));
  state->location[sizeof(state->location) - 1] = '\0';
}

static void mh_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct mh_mailstorage * mh_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_MH);

  mh_data = storage->sto_data;
  
  strncpy(state->location, mh_data->mh_pathname, sizeof(state->location));
  state->location[sizeof(state->location) - 1] = '\0';
}

static void maildir_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct maildir_mailstorage * maildir_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_MAILDIR);

  maildir_data = storage->sto_data;
  
  strncpy(state->location, maildir_data->md_pathname, sizeof(state->location));
  state->location[sizeof(state->location) - 1] = '\0';
}

static void db_storage_set(struct etpan_subapp * app,
    struct mailstorage * storage)
{
  struct app_state * state;
  struct db_mailstorage * db_data;
  
  state = app->data;
  
  state->storage_type = get_storage_index(STORAGE_TYPE_DB);

  db_data = storage->sto_data;
  
  strncpy(state->location, db_data->db_pathname, sizeof(state->location));
  state->location[sizeof(state->location) - 1] = '\0';
}


void etpan_storage_editor_app_set(struct etpan_subapp * app,
    struct mailstorage * storage,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  int type;
  struct app_state * state;
  
  state = app->data;
  
  etpan_storage_editor_app_flush(app);
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  if (storage == NULL)
    return;
  
  type = etpan_cfg_storage_get_type(storage->sto_driver->sto_name);
  
  if (type == -1)
    return;
  
  if (storage->sto_id != NULL) {
    strncpy(state->name, storage->sto_id, sizeof(state->name));
    state->name[sizeof(state->name) - 1] = '\0';
  }
  
  switch (type) {
  case STORAGE_TYPE_IMAP:
    imap_storage_set(app, storage);
    break;

  case STORAGE_TYPE_POP3:
    pop3_storage_set(app, storage);
    break;

  case STORAGE_TYPE_NNTP:
    nntp_storage_set(app, storage);
    break;

  case STORAGE_TYPE_MH:
    mh_storage_set(app, storage);
    break;

  case STORAGE_TYPE_MBOX:
    mbox_storage_set(app, storage);
    break;

  case STORAGE_TYPE_MAILDIR:
    maildir_storage_set(app, storage);
    break;

  case STORAGE_TYPE_DB:
    db_storage_set(app, storage);
    break;
    
  default:
    break;
  }
}



/* extracted and adapted from etpan-cfg-storage.c - begin */

static struct mailstorage * get_mh_storage(char * id, char * location)
{
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = mh_mailstorage_init(storage, location, 1,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_mbox_storage(char * id, char * location)
{
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = mbox_mailstorage_init(storage, location, 1,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_maildir_storage(char * id, char * location)
{
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }

  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);

  r = maildir_mailstorage_init(storage, location, 1,
      cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_db_storage(char * id, char * location)
{
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }

  r = db_mailstorage_init(storage, location);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_nntp_storage(char * id, char * hostname,
    int port, char * command, int connection_type, int auth_type,
    char * login, char * password)
{
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }
  
  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
  r = nntp_mailstorage_init(storage, hostname, port,
      command,
      connection_type,
      auth_type, login, password,
      1, cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_pop3_storage(char * id, char * hostname,
    int port, char * command, int connection_type, int auth_type,
    char * login, char * password)
{
  char cache_directory[PATH_MAX];
  char flags_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }
  
  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  snprintf(flags_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_FLAGS_PATH, id);
  
  r = pop3_mailstorage_init(storage, hostname, port,
      command,
      connection_type,
      auth_type, login, password,
      1, cache_directory, flags_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

static struct mailstorage * get_imap_storage(char * id, char * hostname,
    int port, char * command, int connection_type, int auth_type,
    char * login, char * password)
{
  char cache_directory[PATH_MAX];
  struct mailstorage * storage;
  int r;
  
  storage =  mailstorage_new(id);
  if (storage == NULL) {
    goto err;
  }
  
  snprintf(cache_directory,
      PATH_MAX, "%s/%s/%s", etpan_get_home_dir(), ETPAN_CACHE_PATH, id);
  
  r = imap_mailstorage_init(storage, hostname, port,
      command,
      connection_type,
      auth_type, login, password,
      1, cache_directory);
  if (r != NO_ERROR) {
    goto free;
  }
  
  return storage;
  
 free:
  mailstorage_free(storage);
 err:
  return NULL;
}

/* extracted and adapted from etpan-cfg-storage.c - end */

struct mailstorage *
etpan_storage_editor_app_get_storage(struct etpan_subapp * app)
{
  struct app_state * state;
  int type;
  struct mailstorage * storage;
  char path[PATH_MAX];
  
  state = app->data;
  
  type = storage_type_tab[state->storage_type].type;
  
  storage = NULL;
  switch (type) {
  case STORAGE_TYPE_MH:
    storage = get_mh_storage(state->name, state->location);
    break;
    
  case STORAGE_TYPE_MBOX:
    storage = get_mbox_storage(state->name, state->location);
    break;

  case STORAGE_TYPE_MAILDIR:
    storage = get_maildir_storage(state->name, state->location);
    break;

  case STORAGE_TYPE_DB:
    storage = get_db_storage(state->name, state->location);
    break;

  case STORAGE_TYPE_NNTP:
    storage = get_nntp_storage(state->name, state->hostname, state->port,
        state->command, state->connection_type, state->auth_type,
        state->login, state->password);
    break;

  case STORAGE_TYPE_POP3:
    storage = get_pop3_storage(state->name, state->hostname, state->port,
        state->command, state->connection_type, state->auth_type,
        state->login, state->password);
    break;

  case STORAGE_TYPE_IMAP:
    storage = get_imap_storage(state->name, state->hostname, state->port,
        state->command, state->connection_type, state->auth_type,
        state->login, state->password);
    break;
  }
  
  if (storage == NULL)
    return NULL;
  
  snprintf(path, PATH_MAX, "%s/%s/%s",
      etpan_get_home_dir(), ETPAN_CACHE_PATH, storage->sto_id);
  etpan_mkdir(path);
    
  snprintf(path, PATH_MAX, "%s/%s/%s",
      etpan_get_home_dir(), ETPAN_FLAGS_PATH, storage->sto_id);
  etpan_mkdir(path);
  
  return storage;
}

static void edit_location(struct etpan_subapp * app)
{
  char * default_filename;
  struct app_state * state;
  char current_dir[PATH_MAX];
  int type;
  int file_type;
  
  state = app->data;
  
  default_filename = NULL;
        
  if (state->location[0] == '\0') {
    if (getcwd(current_dir, sizeof(current_dir)) != NULL) {
      strncpy(state->location, current_dir, sizeof(state->location));
      state->location[sizeof(state->location) - 1] = '\0';
    }
  }

  type = storage_type_tab[state->storage_type].type;
  
  file_type = FILENAME_TYPE_REGULAR;
  switch (type) {
  case STORAGE_TYPE_MH:
    file_type = FILENAME_TYPE_MH;
    break;
    
  case STORAGE_TYPE_MAILDIR:
    file_type = FILENAME_TYPE_MAILDIR;
    break;
  }
  
  etpan_cfg_edit_filename_with_type(app,  "Enter path of the mailbox: ",
      state->location, sizeof(state->location), file_type);
}

static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit) {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_EDIT_VALID, state->upcall_data);
  }
  else {
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_STORAGE_EDIT_CANCEL, state->upcall_data);
  }
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "save the folder (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "configuration - not enough memory"));
}

#define HELP_TEXT \
"\
Help for storage properties\n\
---------------------------\n\
\n\
This application will let you edit the properties of a storage.\n\
\n\
- up, down\n\
  arrow keys : move cursor\n\
\n\
- left, right\n\
  arrow keys : change value\n\
- d          : clear selected value \n\
- [Enter]    : edit value\n\
\n\
- y          : finished edition of properties\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
