/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-search-input.c,v 1.8 2003/11/24 20:04:20 hoa Exp $
 */

#include "etpan-search-input.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-input-common.h"
#include "etpan-help-viewer.h"
#include <stdlib.h>

static void handle_key(struct etpan_subapp * app, int key);
static void handle_resize(struct etpan_subapp * app);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_search_input_app_driver = {
  .name = "search-input",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = handle_resize,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_input_common_app_state common_state;
};


static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_display(app, &state->common_state, w);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_input_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_search_input_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_search_input_app_driver);
}

char * etpan_search_input_get_value(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_input_common_get_value(app, &state->common_state);
}

int etpan_search_input_set(struct etpan_subapp * app,
    char * prompt, size_t size,
    char * default_str, int flags,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_input_common_set(app, &state->common_state,
      prompt, size, default_str, flags, upcall, data);
}

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  switch (key) {
  case KEY_F(1):
    show_help(app);
    break;
  }
  
  etpan_input_common_handle_key(app, &state->common_state, key);
}

static void handle_resize(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_handle_resize(app, &state->common_state);
}

static int display_init(struct etpan_subapp * app)
{
  app->show_cursor = 1;
  etpan_subapp_handle_resize(app);
  
  return NO_ERROR;
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_done(&state->common_state);
  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_leave(app, &state->common_state, new_app);
}

#define HELP_TEXT \
"\
Help for input\n\
--------------\n\
\n\
This application will let you enter an information.\n\
\n\
- [Enter]    : edition is finished\n\
- Ctrl-G     : cancel\n\
\n\
- F1         : help\n\
- Ctrl-L     : Console log\n\
\n\
(F1, ? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
