#include <unistd.h>
#include <stdlib.h>
#include "jack.h"
#include <qapplication.h>
#include "mainwnd.h"
#include <qpainter.h>
#include <qpixmap.h>
#include <sys/mman.h>
#include "pump.h"
#include "jacklow.h"
#include "effect.h"
#include "config.h"
#include "frame.h"
#include "jacklow.h"

#define MAX_AGC_DELTA 0.0005l

StreamSource::StreamSource(QWidget *owner)
{
	max_agc=10.0l;
	s_agc=1.0;
	app_window=(MainWnd *)owner;
	max_frames=0L;
	frame_pos=0L;
	player=NULL;
	play_samples=0L;
	playing=0;
	pthread_mutex_init(&control_lock, NULL);
	wavelets=0;
	max_wavelet_value=0;
	wave=NULL;
	wave_source=NULL;
	tid=0;
	pitch=0.0;
	app_window->get_pump()->addStream(this);
}

StreamSource::~StreamSource()
{
	stop();
	if(player)
		player->set_stream(NULL);
	if(tid)
	{
		pthread_cancel(tid);
		pthread_join(tid, NULL);
	}
	if(wave)
	{
		int i;
		for(i=0;i<wavelets;i++)
			if(wave[i])
				delete wave[i];
		delete wave;
	}
	tid=0;
	app_window->get_pump()->removeStream(this);
	if(wave_source)
		delete wave_source;
}

int StreamSource::is_playing()
{
	return playing;
}

int StreamSource::get_buffer(char *buffer, int max)
{
	return 0;
}

unsigned long StreamSource::frame_position()
{
	return frame_pos;
}

unsigned long StreamSource::total_frames()
{
	return max_frames;
}

int StreamSource::play(unsigned long start_frame, unsigned long frames, bool loop)
{
	return -1;
}

void StreamSource::exit_loop(void)
{
}

int StreamSource::stop()
{
	if(player)
		player->buffer->reset();
	return 0;
}

int StreamSource::attach(QString name)
{
	frame_pos=0L;
	return -1;
}

int StreamSource::detach()
{
	stop();

	frame_pos=0L;
	max_frames=0;

	return 0;
}

void StreamSource::bump_frame_position(long by)
{
	if(!by)
		return;
	if(loop_play)
		return;
	unsigned long oldpos=frame_pos;

	frame_pos+=by;
	if(frame_pos < 0)
		frame_pos=0;

	if(frame_pos > max_frames)
		frame_pos=max_frames;
	
	if(frame_pos == oldpos)
		return;
	
	if(playing)
	{
		emit frame_changed(frame_pos);
		app_window->update_display();
	}
}

JackPlayer::JackPlayer(Client *c, int numbase)
{
	ga=1.0;
	s_agc=1.0;
	total_consumed=0;
	client=c;
	att=1.0;
	source=NULL;
	sample_count=0;
	pitch=0.0;
	resample=1.0;

	buffer=new RingBuffer(65536*2);

	QString left_name;
	QString right_name;
	left_name.sprintf("player_%d_left", numbase);
	right_name.sprintf("player_%d_right", numbase);
	left=new Port(c, left_name, -1, Port::Output);
	right=new Port(c, right_name, -1, Port::Output);

	QString play_left, play_right;

	c->addProcessor(this);
	tone_value=1.7;

	int err;

	src_state=src_callback_new(player_callback, SRC_SINC_FASTEST, 2, &err, this);
}

JackPlayer::~JackPlayer()
{
	client->removeProcessor(this);
	delete left;
	delete right;

	if(src_state)
		src_delete(src_state);

	if(source)
	{
		source->stop();
		source->player=NULL;
		source=NULL;
	}
	delete buffer;
}

void JackPlayer::set_stream(StreamSource *stream)
{
	source=stream;
	if(source)
	{
		int sr=source->sample_rate;
		int tr=frame->client()->rate();
		resample=(double)tr/(double)sr;

		source->player=this;
		source->set_pitch(source->pitch);
	}
}

int JackPlayer::process(jack_nframes_t nsamples, void *arg)
{
	JackPlayer *player=this;
    unsigned int sample_off=0;
	int samples_consumed=0;
	unsigned int total_samples=0;
	float copy_buf[16384];

    jack_default_audio_sample_t *s_left=(jack_default_audio_sample_t *) left->buffer (nsamples);
    jack_default_audio_sample_t *s_right=(jack_default_audio_sample_t *) right->buffer (nsamples);

//	total_samples=player->buffer->read((char *)player->frame_buffer, nsamples*sizeof(float)*2)/(sizeof(float)*2);

	double pitch_factor=1.0+(-pitch / 100.0);
	total_samples=src_callback_read(src_state, resample*pitch_factor, nsamples, copy_buf);
	sample_off=0;

	while(sample_off < total_samples)
	{
		s_left[sample_off]=(jack_default_audio_sample_t)copy_buf[sample_off*2];
		if(s_left[sample_off] > 1.0)
			s_left[sample_off]=1.0;
		s_right[sample_off]=(jack_default_audio_sample_t)copy_buf[sample_off*2+1];
		if(s_right[sample_off] > 1.0)
			s_right[sample_off]=1.0;
		sample_off++;

		samples_consumed++;
	}

	while(sample_off < nsamples)
	{
		s_left[sample_off]=0.0;
		s_right[sample_off]=0.0;
		sample_off++;
	}

	if(buffer->write_space() >= 2048*sizeof(float)*2)
	{
		if(player && player->source)
			player->source->get_data();
	}

	QValueList<Effect *>::Iterator ei;
	for(ei=effects.begin();ei!=effects.end();++ei)
	{
		(*ei)->left()->setBuffer(s_left);
		(*ei)->right()->setBuffer(s_right);
		(*ei)->process(nsamples);
	}
	unsigned long i;
	for(i=0;i<nsamples;i++)
	{
		s_left[i]*=(att*ga*s_agc);
		s_right[i]*=(att*ga*s_agc);
	}
	for(ei=postfader_effects.begin();ei!=postfader_effects.end();++ei)
	{
		(*ei)->left()->setBuffer(s_left);
		(*ei)->right()->setBuffer(s_right);
		(*ei)->process(nsamples);
	}
    return 0;
}

void JackPlayer::set_pitch_percent(double p)
{
	pitch=p;
}

void StreamSource::set_pitch(double newpitch)
{
	if(newpitch < -50.0 || newpitch > 50.0)
		return;
	pitch=newpitch;
	if(player)
		player->set_pitch_percent(pitch);
}

void JackPlayer::set_tone_value(double val)
{
	if(val < 0.5 || val > 2.0)
		return;

	tone_value=val;
}

void JackPlayer::setGain(double g)
{
	if(g < 0.0)
		g=0.0;
	if(g > 2.0)
		g=2.0;
	ga=g;
}

double JackPlayer::gain()
{
	return ga;
}

void JackPlayer::setAttenuation(double a)
{
	if(a > 1.0)
		a=1.0;
	if(a < 0.0)
		a=0.0;
	att=a;
}

double JackPlayer::attenuation()
{
	return att;
}

void JackPlayer::addEffect(Effect *e)
{
	if(effects.find(e) == effects.end())
		effects.append(e);
}

void JackPlayer::removeEffect(Effect *e)
{
	QValueList<Effect *>::Iterator it=effects.find(e);
	if(it == effects.end())
		return;
	effects.remove(it);
}

void JackPlayer::addPostFaderEffect(Effect *e)
{
	if(postfader_effects.find(e) == postfader_effects.end())
		postfader_effects.append(e);
}

void JackPlayer::removePostFaderEffect(Effect *e)
{
	QValueList<Effect *>::Iterator it=postfader_effects.find(e);
	if(it == postfader_effects.end())
		return;
	postfader_effects.remove(it);
}

bool StreamSource::event(QEvent *ev)
{
	emit play_done();
	return true;
}

void StreamSource::generate_wave(void)
{
	if(!max_frames)
		return;
	if(wave)
		return;
	if(filename == QString::null)
	{
		printf("generate_wave called while filename is NULL\n");
		return;
	}
	wave_source=get_source();
	int frames;

	wave=new Wavelet*[(frames=(wave_source->total_frames()+32767)/32768)];
	memset((char *)wave, 0, sizeof(Wavelet *)*frames);
	wavelets=frames;

}

StreamSource *StreamSource::get_source(void)
{
	StreamSource *new_source=new StreamSource(app_window);
	new_source->attach(filename);
	return new_source;
}

void StreamSource::get_wave_pixmap(QPixmap *pm, int display_width)
{
	int frame_position=frame_pos;
	if(app_window->is_seeking())
		frame_position=app_window->transport_position();
	int pixel_pos=frame_position/512-display_width/2;
	int w=pixel_pos/64;
	int pixel_offset=pixel_pos-w*64;
	int pos=0;
	int peak_w=frame_position/512/64;
	int peak_offset=frame_position/512-peak_w*64;

	if(peak_w < 0 || peak_w >= wavelets)
		peak_value=0;
	else
	{
		if(wave && !wave[peak_w])
		{
			wave[peak_w]=new Wavelet(wave_source, peak_w*32768, 32768);
			wave_source->play(peak_w*32768, 32768);
			wave[peak_w]->load();
			usleep(50000);
			if(max_wavelet_value < wave[peak_w]->max_wavelet_value)
				max_wavelet_value=wave[peak_w]->max_wavelet_value;
		}

		if(wave && wave[peak_w] && wave[peak_w]->pixmap)
		{
			peak_value=wave[peak_w]->pixmap[peak_offset];
			if(peak_value < 0)
				peak_value=-peak_value;
		}
		else
			peak_value=0;
	}

	QPainter p;

	QPen wap=QColor(36, 152, 255);
	
	p.begin((const QPaintDevice *)pm, (const QWidget *)app_window);
	p.setPen(wap);

	while(pos-pixel_offset < display_width)
	{
		if(w < 0)
		{
			pos+=64;
			w++;
			continue;
		}
		if(w >= wavelets)
			break;

		if(!wave[w])
		{
			wave[w]=new Wavelet(wave_source, w*32768, 32768);
			wave_source->play(w*32768, 32768);
			wave[w]->load();
			usleep(50000);
			if(max_wavelet_value < wave[w]->max_wavelet_value)
				max_wavelet_value=wave[w]->max_wavelet_value;
		}
		if(wave[w] && wave[w]->pixmap)
		{
			int j;

			for(j=0;j<64;j++)
			{
				p.drawPoint(pos-pixel_offset+j, 15-(wave[w]->pixmap[j]));
				p.moveTo(pos-pixel_offset+j, 15-(wave[w]->pixmap[j]));
				p.lineTo(pos-pixel_offset+j, 15+(wave[w]->pixmap[j]));
			}
		}
		pos+=64;
		w++;
	}
	QPen rp(QColor(255, 0, 0));
	p.setPen(rp);
	p.moveTo(display_width/2, 0);
	p.lineTo(display_width/2, pm->height());
}

void StreamSource::set_transport_position(unsigned long pos)
{
	frame_pos=pos;
//	app_window->update_display();
}

void StreamSource::get_data(void)
{
	app_window->get_pump()->work();
}

void StreamSource::prefetch()
{
}

unsigned char **StreamSource::get_frame()
{
	return 0;
}

void StreamSource::lock_frame()
{
}

void StreamSource::unlock_frame()
{
}

bool StreamSource::has_video()
{
	return false;
}

void StreamSource::decode_video()
{
}

void StreamSource::set_preview_framebuffer(int width, int height, unsigned char **ext_framebuf)
{
}

bool StreamSource::writeMetadata(const char *file)
{
    if(!file || !*file)
        return false;

    if(m_artist == "" && m_title == "")
        return false;

    FILE *fp=fopen(file, "w");
    if(fp)
    {
        fprintf(fp, "ARTIST=%s\n", (const char *)m_artist);
        fprintf(fp, "TITLE=%s\n", (const char *)m_title);
        fclose(fp);
        return true;
    }
    return false;
}

QString StreamSource::artist()
{
	return m_artist;
}

QString StreamSource::title()
{
	return m_title;
}

QString StreamSource::display_info()
{
	QString info="";

	if(m_artist)
	{
		info=m_artist;
		if(m_title)
			info=info+" - ";
	}
	if(m_title)
		info=info+m_title;
	return info;
}

void JackPlayer::setAgc(double agc)
{
	s_agc=agc;
}

double JackPlayer::agc()
{
	return s_agc;
}

void StreamSource::calcAgc(char *buf, int max)
{
	int high=0;
	int i;

	for(i=0;i<max;i++)
		if(*((short *)buf+i) > high)
			high=*((short *)buf+i);
	
	if(high < 5000)
		return;

	float val=(float)high/32767.0f;
	float tagc=1.0f/val;

	if(s_agc < tagc)
	{
		if(s_agc < max_agc)
			s_agc+=MAX_AGC_DELTA;
		if(s_agc > max_agc)
			s_agc=max_agc;
	}
	else
		max_agc=tagc;
	
	if(s_agc > max_agc)
		s_agc-=MAX_AGC_DELTA;

//	printf("High %d max %lf Agc targ %f, agc %lf\n", high, max_agc, tagc, s_agc);
	emit agc_change((double)s_agc);
}

bool StreamSource::can_do_multiple_reads()
{
	return true;
}

long JackPlayer::player_callback(void *thisp, float **data)
{
	JackPlayer *player=(JackPlayer *)thisp;

	if(!player)
		return 0L;
	*data=player->frame_buffer;

	if(player->buffer->read_space() < 1024*sizeof(float)*2 || !player->source)
	{
		memset(player->frame_buffer, 0, 1024*sizeof(float)*2);
		return 1024;
	}

	player->total_consumed+=1024;
	if(player->total_consumed >= 2048)
	{
		if(player->source)
			player->source->bump_frame_position(player->total_consumed);
		player->total_consumed=0;
	}

	player->sample_count=player->buffer->read((char *)player->frame_buffer, 1024*sizeof(float)*2)/(sizeof(float)*2);

	return player->sample_count;
}
