# bug-triage -- bug triage and forward tool.
# Copyright (C) 2007  Gustavo R. Montesino
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# This module is responsible for handling the integration between Debian 
# packages and upstream bug tracking systems.

import ConfigParser
import os, os.path
import sys

import btstype
from bt_exceptions import *

class Package:
    """Represents a debian package and it's upstream BTS data"""
    
    def get_package(self):
        return self.package

    def set_package(self, package):
        self.package = package

    def get_type(self):
        return self.type

    def get_type_str(self):
        return btstype.STRINGS[self.type]

    def set_type(self, type):
        if type < len(btstype.STRINGS):
            self.type = type
        else:
            raise InvalidBTSError, "Invalid upstream type: %s" % type

    def set_type_str(self, type):
        if type.lower() in btstype.STRINGS:
            self.type = btstype.STRINGS.index(type.lower())
        else:
            raise InvalidBTSError, "Invalid upstream type: %s" % type

    def get_data(self):
        return self.data

    def set_data(self, data):
        self.data = data

# Our list of packages. Keep this separated to make sure a user won't get
# global entries in his file.
DefaultPackages = ConfigParser.ConfigParser()
DefaultPackages.read(["/etc/bug-triage/packages.conf"])
for dir in sys.path:
    file = dir + "config/data/packages.conf"
    if os.access(file, os.R_OK):
        DefaultPackages.read([file])
        break

# User's list of packages.
UserFile = os.path.expanduser("~/.bug-triage/packages.conf")
UserPackages = ConfigParser.ConfigParser()
UserPackages.read([UserFile])

def get(packagename):
    """Returns the Upstream of a package"""

    # Check the user's list first -- allow the user to override stuff
    if UserPackages.has_section(packagename):
        config = UserPackages
    elif DefaultPackages.has_section(packagename):
        config = DefaultPackages
    else:
        return False

    package = Package()
    package.set_package(packagename)
    package.set_type_str(config.get(packagename, "type"))

    data = { }
    for option in config.options(packagename):
        if option != "type":
            data[option] = config.get(packagename, option)
    package.set_data(data)

    return package

def get_all():
    """Returns a list with all package data available

    If a package is on both (user and default) files, the entry from
    the user file takes preference."""

    packages = []

    for pkgname in UserPackages.sections():
        current = get(pkgname)
        packages.append(current)

    for pkgname in DefaultPackages.sections():
        if not UserPackages.has_section(pkgname):
            current = get(pkgname)
            packages.append(current)

    return packages

def write_user_file():
    """Writes any changes to the user packages config file"""
 
    if not os.path.isdir(os.path.dirname(UserFile)):
        os.mkdir(os.path.dirname(UserFile))
    
    f = open(UserFile, 'w')
    UserPackages.write(f)

def set(package, type, data):
    """Writes a package settings on the user file"""

    if not UserPackages.has_section(package):
        UserPackages.add_section(package)
    else:
        for option in UserPackages.options(package):
            UserPackages.remove_option(package,option)

    UserPackages.set(package, "type", btstype.STRINGS[type])
    for option, value in data.iteritems():
        UserPackages.set(package, option, value)

    write_user_file()

def remove(package):
    """Remove a package settings"""

    if UserPackages.remove_section(package):
        write_user_file()
    else:
        if DefaultPackages.has_section(package):
            raise InvalidPackage("Can't remove a package on the default \
packages file; if the information is wrong please override \
it and file a bug report.")
        else:
            raise InvalidPackage("Couldn't find package: %s" % package)

# vim: tabstop=4 expandtab shiftwidth=4
