/* Workspace.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Logger;

import org.grinvin.io.Subsystem;
import org.grinvin.io.WorkspaceSaver;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;

/**
 * Handles session persistence. The singleton object of this type
 * allows sessions to be persisted to a workspace directory on disk.
 */
public class Workspace {
    
    //
    private static final Logger LOGGER
            = Logger.getLogger("org.grinvin.workspace", "org.grinvin.workspace.resources");
    
    //
    private File location;
    
    //
    public List<Subsystem> subsystems;
    
    //
    private List<WorkspaceListener> listeners;
    
    //
    private Workspace() {
        this.location = null;
        this.subsystems = new ArrayList<Subsystem>();
        listeners = new ArrayList<WorkspaceListener>();
    }
    
    //
    private static final Workspace SINGLETON = new Workspace();
    
    /**
     * Get the singleton instance of this class
     * @return the single instance of this class
     */
    public static Workspace getInstance() {
        return SINGLETON;
    }
    
    //
    public File getLocation() {
        return location;
    }
    
    // changes location and remembers it as a preference
    public void setLocation(File location) {
        this.location = location;
        GrinvinPreferences.getInstance().setStringPreference(
                Preference.WORKSPACE_LOCATION,
                location.getAbsolutePath());
        fireLocationChanged();
    }
    
    //
    public void addWorkspaceListener(WorkspaceListener listener) {
        listeners.add(listener);
    }
    
    //
    public void removeWorkspaceListener(WorkspaceListener listener) {
        listeners.remove(listener);
    }
    
    //
    public void fireLocationChanged() {
        for (WorkspaceListener listener : listeners)
            listener.locationChanged();
    }
    
    /**
     * Register (the wrapper) of a subsystem. The subsystem that is registered
     * first is also loaded and initialized first, but disposed and saved last.
     * If a subsystem A depends on a subsystem B, then B should be registered
     * before A.
     * @throws IllegalStateException when called after
     * a workspace location has already been set
     */
    public void registerSubsystem(Subsystem subsystem) {
        if (location != null)
            throw new IllegalStateException("Cannot register a subsystem in a running system");
        subsystems.add(subsystem);
    }
    
    //
    public void dispose() {
        if (this.location != null) {
            // in reverse order
            for (ListIterator<Subsystem> iter = subsystems.listIterator(subsystems.size());
            iter.hasPrevious(); )
                iter.previous().dispose();
        }
        this.location = null;
    }
    
    /**
     * Initialize all subsystems and perform an initial save into the
     * given workspace directory. If there is an active workspace already,
     * it is disposed of first. If the new directory already exists,
     * all its contents are removed.
     */
    public void newWorkspace(File location) throws WorkspaceException {
        dispose();
        // in order of registration
        for (Subsystem subsystem : subsystems)
            subsystem.init();
        WorkspaceSaver.saveAs(this, location);
    }

}
