/* WorksheetRename.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.actions;


import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.grinvin.worksheet.WorksheetModel;
import org.grinvin.worksheet.WorksheetWindowListModel;
import org.grinvin.main.GrinvinWindow;
import org.grinvin.main.GrinvinWindow.GrinvinWindowComponentType;


/**
 * GUI command which allows the user to rename the currently selected
 * worksheet.<p>
 * Acts upon a array list model and a selection model stored as attributes in
 * an attributed command manager.
 */
public class WorksheetRename extends AbstractAction implements ListSelectionListener, ChangeListener {

    //
    private final WorksheetWindowListModel list;
    
    //
    private final ListSelectionModel selectionModel;
    
    //
    private final GrinvinWindow window;

    /**
     * Construct an action command of this type.
     */
    public WorksheetRename(GrinvinWindow window) {
        this.window = window;
        this.list = window.getWorksheetListModel().getWorksheetWindowListModel();
        this.selectionModel = window.getWorksheetListSelectionModel();
        this.selectionModel.addListSelectionListener(this);
        this.window.addTabsChangeListener(this);
        int index = selectionModel.getMinSelectionIndex();
        setEnabled(index >= 0 && index == selectionModel.getMaxSelectionIndex() && window.getSelectedTab().equals(GrinvinWindowComponentType.WORKSHEETS));
    }

    /**
     * Ask the user to rename the currently selected graph list element.
     */
    public void actionPerformed(ActionEvent e) {
        // TODO: delegate some of this to WorksheetListModel, add method getSelectedWindows?

        int index = selectionModel.getMinSelectionIndex();
        if (index < 0) {
            return;
        } // none selected
        if (index != selectionModel.getMaxSelectionIndex()) {
            return;
        } // more than one selection

        WorksheetModel model = list.get(index).getModel();
        if (model.isNameEditable()) {
            WorksheetRenamePanel panel = new WorksheetRenamePanel();
            panel.showDialog(model, window);
        }
    }

    public void valueChanged(ListSelectionEvent e) {
        int index = selectionModel.getMinSelectionIndex();
        setEnabled(index >= 0 && index == selectionModel.getMaxSelectionIndex() && window.getSelectedTab().equals(GrinvinWindowComponentType.WORKSHEETS));
    }

    public void stateChanged(ChangeEvent e) {
        int index = selectionModel.getMinSelectionIndex();
        setEnabled(index >= 0 && index == selectionModel.getMaxSelectionIndex() && window.getSelectedTab().equals(GrinvinWindowComponentType.WORKSHEETS));
    }
}
