/* GridLines.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.guides;

import java.awt.Graphics2D;
import java.awt.geom.Line2D;

/**
 * Guides based on a square grid within the -1.0 - 1.0 rectangle. The number of
 * horizontal and vertical grid lines is configurable.
 */
public class GridLines extends AbstractGuides {
    
    //
    private int numberOfRows;
    
    //
    private int numberOfColumns;
    
    /**
     * Set the number of divisions used by this grid.
     * @param numberOfRows Number of rows in the grid, i.e., one less than the
     * number of horizontal grid lines.
     * @param numberOfColumns Number of columns in the grid, i.e., one less than the
     * number of vertical grid lines.
     *
     */
    public void setNumberOfDivisions (int numberOfRows, int numberOfColumns) {
        if (numberOfRows < 1 || numberOfColumns < 1)
            throw new IllegalArgumentException ("Illegal number of grid divisions");
        this.numberOfRows = numberOfRows;
        this.numberOfColumns = numberOfColumns;
    }
    
    /**
     * Create a new object of this type.
     * @param numberOfRows Number of rows in the grid, i.e., one less than the
     * number of horizontal grid lines.
     * @param numberOfColumns Number of columns in the grid, i.e., one less than the
     * number of vertical grid lines.
     */
    public GridLines (int numberOfRows, int numberOfColumns) {
        this.numberOfRows = numberOfRows;
        this.numberOfColumns = numberOfColumns;
    }
    
    /**
     * Paint a grid with the requested number of subdivisions
     */
    public void paint (Graphics2D g2, double scale, double maxX, double maxY) {
        g2.setStroke (GUIDE_STROKE);
        g2.setPaint (GUIDE_COLOR);
        
        double halfSize = scale + 2;
        
        // horizontal lines
        Line2D line = new Line2D.Double (); // reusable line
        for (int i = 0; i <= numberOfRows; i++) {
            double y = (-1.0 + 2.0*i/numberOfRows)*scale;
            line.setLine (-halfSize, y, halfSize, y);
            g2.draw (line);
        }
        
        // vertical lines
        for (int i = 0; i <= numberOfColumns; i++) {
            double x = (-1.0 + 2.0*i/numberOfColumns)*scale;
            line.setLine (x, -halfSize, x, halfSize);
            g2.draw (line);
        }
        
    }
    
    /**
     * Snap coordinates to the closest grid position.
     */
    public void snap (double[] coords) {
        // snap X-coordinate
        double x = coords[0];
        if (x < -1.0)
            coords[0] = -1.0;
        else if (x > 1.0)
            coords[0] = 1.0;
        else {
            x = Math.floor ((x+1.0)*numberOfColumns/2.0 + 0.5);
            coords[0] = 2.0*x/numberOfColumns -1.0;
        }
        // snap Y-coordinate
        double y = coords[1];
        if (y < -1.0)
            coords[1] = -1.0;
        else if (y > 1.0)
            coords[1] = 1.0;
        else {
            y = Math.floor ((y+1.0)*numberOfRows/2.0 + 0.5);
            coords[1] = 2.0*y/numberOfRows -1.0;
        }
    }
    
}
