/* Compound.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.expr;

/**
 * Compound expression, i.e., an expression containing one or several 'sub-expressions'.
 */
public class Compound implements Expression {
    
    //
    private final Operator operator;

    /**
     * Get the operator for this expression.
     */
    public Operator getOperator() {
        return operator;
    }
    
    /**
     * The number of argument children for this compound expression.
     */
    public int getNumberOfChildren () {
        return children.length;
    }
    
    /**
     * Return the child with the given index.
     */
    public Expression getChild (int index) {
        return children[index];
    }
    
    //
    private final Expression[] children;
    
    //
    public void accept(Expression.Visitor visitor) {
        visitor.visitCompound(this);
    }
    
    /**
     * Create an expression which applies the given operator to the
     * given array of arguments.
     */
    public Compound (Operator operator, Expression[] children) {
        assert operator.getNumberOfArguments () == children.length;
        this.operator = operator;
        this.children = children.clone();
    }
    
    /**
     * Create an expression which applies the given operator to the
     * given argument.
     */
    public Compound (Operator operator, Expression arg) {
        assert operator.getNumberOfArguments () == 1;
        this.operator = operator;
        this.children = new Expression[] { arg } ;
    }
    
    /**
     * Create an expression which applies the given operator to the
     * two given arguments.
     */
    public Compound (Operator operator, Expression arg1, Expression arg2) {
        assert operator.getNumberOfArguments () == 2;
        this.operator = operator;
        this.children = new Expression[] { arg1, arg2 } ;
    }
    
}
