/*
    This file is part of SUPPL - the supplemental library for DOS
    Copyright (C) 1996-2000 Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: FREEMCB.C $
   $Locker: ska $	$Name:  $	$State: Exp $

	freeMCB [XXXX]

	Utility to free a MCB. There won't be made any checks whether or not
	the MCB to be freed is possibly in use.

	With the hexadecimal value XXXX, this MCB is freed. The address can
	point either to the MCB itself or a data segment.

	Without the value, all MCB's are displayed and a MCB to be freed
	can be choosen. To address a MCB the *upper case& letter must be
	pressed!

*/

#include <stdio.h>
#include <portable.h>
#include <mcb.h>

#define MAXLINES 20

#define Q_NEXT 0
#define Q_QUIT 1
#define Q_RESTART 2

#ifndef lint
static char const rcsid[] = 
	"$Id: FREEMCB.C 1.2 1996/07/25 06:37:10 ska Exp ska $";
#endif

int lines = 0;
unsigned mcb[MAXLINES];

void freeMCB(unsigned mcbNr)
{	unsigned n;

	if(isMCB(n = mcbNr) || isMCB(n = SEG2MCB(mcbNr))) {
		MCB2SEG(n);		/* mov ax, data_area_of(n) */
		asm {
			mov es, ax
			mov ah, 49h
			int 21h
			rcl al, 1
		}
		if(nargs() & 1) {
			printf("! Free Memory of MCB %04x failed\n", mcbNr);
			exit(255);
		}
		else printf("MCB %04x freed\n", mcbNr);
	}
	else {
		printf("! No MCB %04x\n", mcbNr);
		exit(255);
	}
}

int interaction(void)
{	static char line[] = "\nFree: X - Y, Restart: ', Next Page: Enter, Quit: ESC";
	unsigned c, l;

	if(lines) {
		line[7] = 'A';
		l = line[11] = 'A' - 1 + lines;
		lines = 0;
	}
	else {
		line[7] = line[11] = ' ';
	}

	puts(line);

	do {
		switch(c = kbget()) {
			case 0x1b: return Q_QUIT;			/* Escape */
			case ' ': case '\n': case '\r':	/* next page */
				return Q_NEXT;
			default:
				if(c < 'A' || c > l)
					break;
				freeMCB(mcb[c - 'A']);
			case '\'': return Q_RESTART;			/* restart */
		}
		beep(1000, 100);
	} while(1);
}

void dispName(unsigned char *buf)
{	int i;

	for(i = buf[8] = 0; i < 8; ++i)
		if(buf[i] < ' ') buf[i] = ' ';
	fputs(buf, stdout);
}

int dispMCB(nM(void *)arg, unsigned segm)
{	unsigned char buf[9];
	int i;

	if(!lines) puts("\n\n");

	printf("%c %04x (%04x) ", 'A' + lines, mcb[lines] = segm
	 , peekw(segm, OFF_MCB_SIZE));

	for(i = 0; i < 8; ++i) {
		printf("%02x", buf[i] = peekb(segm, i + 8));
		if((i & 3) == 3) putchar(' ');
	}

	dispName(buf);
	if((i = peekw(segm, OFF_MCB_OWNER)) == 0)
		puts("  Unallocated");
	else if(i <= 8)
		puts("  System memory");
	else {
		if((i = mcb_owner(segm)) && i != segm) {
			fputs("  Owner: ", stdout);
			copy_seg(get_ss(), buf, i, 8, 8);
			dispName(buf);
		}
		putchar('\n');
	}

	return ++lines >= MAXLINES? interaction(): Q_NEXT;
}

main(int argc, char **argv)
{	switch(argc) {
		default:
			printf("Useage: %s [XXXX]\nDisplay and/or free a MCB\n! Too many arguments\n", argv[1]);
			exit(100);
		case 2:
			freeMCB(atox(argv[1]));
			break;
		case 1:
			while((argc = mcb_walk(0, aF(dispMCB), 0)) >= 0) {
				if(argc == Q_NEXT && lines)
					argc = interaction();
				if(argc == Q_QUIT) break;
			}

			if(argc < 0) {
				abort("! MCB chain corrupt\n");
			}
			break;
	}
}
