/*
   Unix SMB/CIFS implementation.
   async dsgetdcname
   Copyright (C) Volker Lendecke 2009

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "includes.h"
#include "winbindd.h"
#include "librpc/gen_ndr/cli_wbint.h"

struct wb_dsgetdcname_state {
	struct netr_DsRGetDCNameInfo *dcinfo;
};

static void wb_dsgetdcname_done(struct tevent_req *subreq);

struct tevent_req *wb_dsgetdcname_send(TALLOC_CTX *mem_ctx,
				       struct tevent_context *ev,
				       const char *domain_name,
				       const struct GUID *domain_guid,
				       const char *site_name,
				       uint32_t flags)
{
	struct tevent_req *req, *subreq;
	struct wb_dsgetdcname_state *state;
	struct winbindd_child *child;
	struct GUID guid;
	struct GUID *guid_ptr = NULL;

	req = tevent_req_create(mem_ctx, &state, struct wb_dsgetdcname_state);
	if (req == NULL) {
		return NULL;
	}

	if (strequal(domain_name, "BUILTIN")
	    || strequal(domain_name, get_global_sam_name())) {
		/*
		 * Two options here: Give back our own address, or say there's
		 * nobody around. Right now opting for the latter, one measure
		 * to prevent the loopback connects. This might change if
		 * needed.
		 */
		tevent_req_nterror(req, NT_STATUS_DOMAIN_CONTROLLER_NOT_FOUND);
		return tevent_req_post(req, ev);
	}

	if (IS_DC) {
		/*
		 * We have to figure out the DC ourselves
		 */
		child = locator_child();
	} else {
		struct winbindd_domain *domain = find_our_domain();
		child = &domain->child;
	}

	if (domain_guid != NULL) {
		/* work around a const issue in rpccli_ autogenerated code */
		guid = *domain_guid;
		guid_ptr = &guid;
	}

	subreq = rpccli_wbint_DsGetDcName_send(
		state, ev, child->rpccli, domain_name, guid_ptr, site_name,
		flags, &state->dcinfo);
	if (tevent_req_nomem(subreq, req)) {
		return tevent_req_post(req, ev);
	}
	tevent_req_set_callback(subreq, wb_dsgetdcname_done, req);
	return req;
}

static void wb_dsgetdcname_done(struct tevent_req *subreq)
{
	struct tevent_req *req = tevent_req_callback_data(
		subreq, struct tevent_req);
	struct wb_dsgetdcname_state *state = tevent_req_data(
		req, struct wb_dsgetdcname_state);
	NTSTATUS status, result;

	status = rpccli_wbint_DsGetDcName_recv(subreq, state, &result);
	TALLOC_FREE(subreq);
	if (!NT_STATUS_IS_OK(status)) {
		tevent_req_nterror(req, status);
		return;
	}
	tevent_req_done(req);
}

NTSTATUS wb_dsgetdcname_recv(struct tevent_req *req, TALLOC_CTX *mem_ctx,
			     struct netr_DsRGetDCNameInfo **pdcinfo)
{
	struct wb_dsgetdcname_state *state = tevent_req_data(
		req, struct wb_dsgetdcname_state);
	NTSTATUS status;

	if (tevent_req_is_nterror(req, &status)) {
		return status;
	}
	*pdcinfo = talloc_move(mem_ctx, &state->dcinfo);
	return NT_STATUS_OK;
}
