/*
 * Copyright 2004 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package sun.jvmstat.monitor.event;

import java.util.Set;
import sun.jvmstat.monitor.MonitoredHost;

/**
 * Provides a description of a change in status of the Java Virtual Machines
 * associated with a MonitoredHost.
 *
 * @author Brian Doherty
 * @since 1.5
 */
public class VmStatusChangeEvent extends HostEvent {

    /**
     * The set of currently active Java Virtual Machines for the MonitoredHost.
     * The set contains an Integer object holding the <em>lvmid</em> for each
     * active Java Virtual Machine on the MonitoredHost. This Set will only
     * contain Integer objects.
     */
    protected Set active;

    /**
     * The set of Java Virtual Machines started on MonitoredHost since the
     * previous event. The set contains an Integer object holding the
     * <em>lvmid</em> for each Java Virtual Machine started on the
     * MonitoredHost. This Set will only contain Integer objects.
     */
    protected Set started;

    /**
     * The set of Java Virtual Machines terminated on MonitoredHost since the
     * previous event. The set contains an Integer object holding the
     * <em>lvmid</em> for each Java Virtual Machine started on the
     * MonitoredHost. This Set will only contain Integer objects.
     */
    protected Set terminated;

    /**
     * Construct a new VmStatusChangeEvent instance.
     *
     * @param host the MonitoredHost that is the source of the event.
     * @param active the set of currently active Java Virtual Machines
     * @param started the set of Java Virtual Machines started since the
     *                last event.
     * @param terminated the set of Java Virtual Machines terminated since
     *                   the last event.
     */
    public VmStatusChangeEvent(MonitoredHost host, Set active,
                               Set started, Set terminated) {
        super(host);
        this.active = active;
        this.started = started;
        this.terminated = terminated;
    }

    /**
     * Return the set of currently active Java Virtual Machines.
     * The set contains an Integer object holding the <em>lvmid</em> for each
     * active Java Virtual Machine on the MonitoredHost.
     *
     * @return Set - a set of Integer objects containing the <em>lvmid</em>
     *               of each active Java Virtual Machine on the host. If
     *               there are no active Java Virtual Machines on the host,
     *               an empty Set is returned.
     */
    public Set getActive() {
        return active;
    }

    /**
     * Return the set of Java Virtual Machines started since the last
     * event notification. The set contains an Integer object holding
     * the <em>lvmid</em> for each Java Virtual Machine started on the
     * MonitoredHost since the last event notification.
     *
     * @return Set - a set of Integer objects containing the <em>lvmid</em>
     *               of each Java Virtual Machine started on the host. If
     *               no Java Virtual Machines were recently started on the
     *               host, an empty Set is returned.
     */
    public Set getStarted() {
        return started;
    }

    /**
     * Return the set of Java Virtual Machines terminated since the last
     * event notification. The set contains an Integer object holding
     * the <em>lvmid</em> for each Java Virtual Machine terminated on the
     * MonitoredHost since the last event notification.
     *
     * @return Set - a set of Integer objects containing the <em>lvmid</em>
     *               of each Java Virtual Machine terminated on the host. If
     *               no Java Virtual Machines were recently terminated on the
     *               host, an empty Set is returned.
     */
    public Set getTerminated() {
        return terminated;
    }
}
