/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id$
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "txlist.h"
#include "payee.h"
#include "category.h"
#include "kbanking.h"

#include <qdatetime.h>
#include <qheader.h>


#define TRANSACTIONLIST2_MAXVALUE ((double)(200000000.00))



TransactionListView2Item::TransactionListView2Item(TransactionListView2 *parent,
                                                   RefPointer<Transaction> t)
:RichTextListViewItem(parent)
,_transaction(t) {
}



TransactionListView2Item::TransactionListView2Item(TransactionListView2 *parent,
                                                   TransactionListView2Item *after,
                                                   RefPointer<Transaction> t)
:RichTextListViewItem(parent, after)
,_transaction(t) {
}



TransactionListView2Item::TransactionListView2Item(const TransactionListView2Item &item)
:RichTextListViewItem(item)
,_transaction(item._transaction) {
}



TransactionListView2Item::~TransactionListView2Item() {
}



RefPointer<Transaction> TransactionListView2Item::getTransaction() {
  return _transaction;
}



void TransactionListView2Item::setup() {
  for (int i=0; i<listView()->columns(); i++)
    setupColumn(i);
  RichTextListViewItem::setup();
}



void TransactionListView2Item::setupColumn(int col) {
  QString tmp;
  const std::list<std::string> sl;
  std::list<std::string>::const_iterator it;
  const GWEN_TIME *ti;
  const AB_VALUE *v;
  std::string pid;
  std::string cid;
  QString pname;
  QString cname;
  TransactionListView2 *lv;
  KBanking *app;

  lv=dynamic_cast<TransactionListView2*>(listView());
  assert(lv);
  app=lv->app();

  if (col!=3 && !text(col).isEmpty())
    // only modify data if we have to
    return;

  /* prepare data */
  switch(col) {
  case 0:
    // date
    tmp="<qt>";
    ti=_transaction.ref().getDate();
    if (!ti)
      ti=_transaction.ref().getValutaDate();
    if (ti) {
      int year, month, day;

      if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year)) {
        QDate t;

        t=QDate(year, month+1, day);
        tmp+=t.toString(Qt::LocalDate);
      }
    }
    tmp+="</qt>";
    setText(col, tmp);
    break;

  case 1:
    // institute
    tmp="<qt>";
    tmp+=QString::fromUtf8(_transaction.ref().getRemoteBankCode().c_str());
    tmp+="</qt>";
    setText(col, tmp);
    break;

  case 2:
    // account
    tmp="<qt>";
    tmp+=QString::fromUtf8(_transaction.ref().getRemoteAccountNumber().c_str());
    tmp+="</qt>";
    setText(col, tmp);
    break;

  case 3:
    tmp="<qt>";
    if (lv->currentItem()==this || !lv->isShortList()) {
      // other name
      for (it=_transaction.ref().getRemoteName().begin();
           it!=_transaction.ref().getRemoteName().end();
           it++) {
        tmp+=QString::fromUtf8((*it).c_str());
        tmp+="<br>";
      } // for

      // description
      for (it=_transaction.ref().getPurpose().begin();
           it!=_transaction.ref().getPurpose().end();
           it++) {
        tmp+="<br>";
        tmp+=QString::fromUtf8((*it).c_str());
      } // for
    }
    else {
      // other name
      if (!_transaction.ref().getRemoteName().empty()) {
        tmp+=QString::fromUtf8(_transaction.ref().getRemoteName().front().c_str());
        tmp+="<br>";
      }
      else if (!_transaction.ref().getPurpose().empty()) {
        tmp+=QString::fromUtf8(_transaction.ref().getPurpose().front().c_str());
        tmp+="<br>";
      }
      else {
        tmp="<br>";
      }
    }
    tmp+="</qt>";
    setText(col, tmp);
    break;

  case 4:
    // type
    tmp="<qt>";
    tmp+=QString::fromUtf8(_transaction.ref().getTransactionText().c_str());
    tmp+=" (";
    tmp+=QString::number(_transaction.ref().getTransactionCode());
    tmp+=")";
    tmp+="</qt>";
    setText(col, tmp);
    break;

  case 5:
    // value
    tmp="<qt> <p align=right>";
    v=_transaction.ref().getValue();
    if (v) {
      if (AB_Value_IsPositive(v))
        tmp+="<font color=\"dark green\">";
      else
        tmp+="<font color=red>";
      tmp+=QString::number(AB_Value_GetValueAsDouble(v), 'f', 2);
      tmp+=" ";
      tmp+=QString::fromUtf8(AB_Value_GetCurrency(v));
      tmp+="</font>";
    }
    tmp+="</p></qt>";
    setText(col, tmp);
    break;

  case 6:
    if (text(col).isEmpty()) {
      // payee and category, payee here
      pid=_transaction.ref().getPayee();
      pname==QWidget::tr("unknown");
      if (!pid.empty()) {
        if (app) {
          Payee *p;
  
          p=app->findPayeeById(pid.c_str());
          if (p)
            pname=QString::fromUtf8(p->name().c_str());
        }
      }
  
      // category
      cid=_transaction.ref().getCategory();
      cname==QWidget::tr("unknown");
      if (!cid.empty()) {
        KBanking *app=lv->app();
        if (app) {
          Category *cat;
  
          cat=app->findCategoryById(cid.c_str());
          if (cat) {
            if (cat->isIncome())
              cname="<font color=\"dark green\">";
            else
              cname="<font color=red>";
            cname+=QString::fromUtf8(cat->getName().c_str());
            cname+="</font>";
          }
        }
      }
  
      tmp="<qt>";
      tmp+=pname;
      if (!cname.isEmpty()) {
        tmp+="<br>";
        tmp+=cname;
      }
      tmp+="</qt>";
      setText(col, tmp);
    }
    break;

  default:
    break;
  } // switch

}



QString TransactionListView2Item::key(int column, bool ascending) const {
  QString result;

  result="";

  switch(column) {
  case 0: { // date
    const GWEN_TIME *ti;

    ti=_transaction.ref().getDate();
    if (!ti)
      ti=_transaction.ref().getValutaDate();
    if (ti) {
      GWEN_BUFFER *tbuf;

      tbuf=GWEN_Buffer_new(0, 32, 0, 1);
      GWEN_Time_toString(ti, "YYYYMMDD-hh:mm", tbuf);
      result=QString(GWEN_Buffer_GetStart(tbuf));
      GWEN_Buffer_free(tbuf);
    }
    else
      result=text(column);
    break;
  }

  case 5: { // value
    const AB_VALUE *v;

    v=_transaction.ref().getValue();
    if (v) {
      const char *s;
      double d;
      char numbuf[32];

      d=TRANSACTIONLIST2_MAXVALUE-AB_Value_GetValueAsDouble(v);
      snprintf(numbuf, sizeof(numbuf), "%012lf.2", d);
      result=QString(numbuf);
      s=AB_Value_GetCurrency(v);
      if (s) {
	result+= " " + QString::fromUtf8(AB_Value_GetCurrency(v));
      }
    }
    else
      result="";
    break;
  }

  default:
    result=text(column);
    break;
  } // switch

  return result;
}



void TransactionListView2Item::redrawItem() {
  for (int i=0; i<listView()->columns(); i++)
    setText(i, "");
  widthChanged();
  repaint();
}








TransactionListView2::TransactionListView2(KBanking *app,
                                           bool shortList,
                                           QWidget *parent,
                                           const char *name)
:RichTextListView(parent, name)
,_app(app)
,_shortList(shortList)
,_lastSessionId(0) {
  _init();
}



TransactionListView2::TransactionListView2(KBanking *app, QWidget *parent, const char *name)
:RichTextListView(parent, name)
,_app(app)
,_shortList(false)
,_lastSessionId(0) {
  _init();
}



TransactionListView2::TransactionListView2(QWidget *parent, const char *name)
:RichTextListView(parent, name)
,_app(0)
,_shortList(false)
,_lastSessionId(0) {
  _init();
}



TransactionListView2::~TransactionListView2() {
}



void TransactionListView2::_init() {
  setAllColumnsShowFocus(true);
  setShowSortIndicator(true);
  setItemMargin(2);
  addColumn(QWidget::tr("Date"),-1);
  addColumn(QWidget::tr("Institute"),-1);
  addColumn(QWidget::tr("Account"),-1);
  addColumn(QWidget::tr("Name/Purpose"),-1);
  addColumn(QWidget::tr("Type"),-1);
  addColumn(QWidget::tr("Value"),-1);
  addColumn(QWidget::tr("Payee"),-1);
  setColumnAlignment(5, Qt::AlignRight);
  header()->setTracking(false);
}



KBanking *TransactionListView2::app() {
  return _app;
}



void TransactionListView2::setApp(KBanking *app) {
  _app=app;
}



void TransactionListView2::addTransaction(RefPointer<Transaction> t) {
  TransactionListView2Item *entry;

  entry=new TransactionListView2Item(this, t);
}



void TransactionListView2::addTransactions(const std::list<RefPointer<Transaction> > &ts) {
  std::list<RefPointer<Transaction> >::const_iterator it;

  for (it=ts.begin(); it!=ts.end(); it++)
    addTransaction(*it);
}



RefPointer<Transaction> TransactionListView2::getCurrentTransaction() {
  TransactionListView2Item *entry;

  entry=dynamic_cast<TransactionListView2Item*>(currentItem());
  if (!entry) {
    DBG_DEBUG(0, "No item selected in list.");
    return 0;
  }
  return entry->getTransaction();
}



std::list<RefPointer<Transaction> > TransactionListView2::getSelectedTransactions() {
  std::list<RefPointer<Transaction> > ts;
  TransactionListView2Item *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      entry=dynamic_cast<TransactionListView2Item*>(it.current());
      if (entry)
        ts.push_back(entry->getTransaction());
    }
  } // for

  return ts;
}



void TransactionListView2::redrawSelected() {
  TransactionListView2Item *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      entry=dynamic_cast<TransactionListView2Item*>(it.current());
      assert(entry);
      entry->redrawItem();
    }
  } // for
}



bool TransactionListView2::isShortList() const {
  return _shortList;
}



void TransactionListView2::setShortList(bool b) {
  _shortList=b;
}



uint32_t TransactionListView2::getLastSessionId() const {
  return _lastSessionId;
}



void TransactionListView2::setLastSessionId(uint32_t sess) {
  _lastSessionId=sess;
}







