//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <cuda/std/iterator>

// reverse_iterator

// template <RandomAccessIterator Iter1, RandomAccessIterator Iter2>
//   requires HasMinus<Iter2, Iter1>
//   constexpr auto operator-(const reverse_iterator<Iter1>& x, const reverse_iterator<Iter2>& y)
//   -> decltype(y.base() - x.base());
//
// constexpr in C++17

#include <cuda/std/iterator>
#include <cuda/std/cstddef>
#include <cuda/std/cassert>

#include "test_macros.h"
#include "test_iterators.h"

template <class It1, class It2>
__host__ __device__
void
test(It1 l, It2 r, cuda::std::ptrdiff_t x)
{
    const cuda::std::reverse_iterator<It1> r1(l);
    const cuda::std::reverse_iterator<It2> r2(r);
    assert((r1 - r2) == x);
}

int main(int, char**)
{
    char s[3] = {0};
    test(random_access_iterator<const char*>(s), random_access_iterator<char*>(s), 0);
    test(random_access_iterator<char*>(s), random_access_iterator<const char*>(s+1), 1);
    test(random_access_iterator<const char*>(s+1), random_access_iterator<char*>(s), -1);
    test(s, s, 0);
    test(s, s+1, 1);
    test(s+1, s, -1);

#if TEST_STD_VER > 14
    {
        constexpr const char *p = "123456789";
        typedef cuda::std::reverse_iterator<const char *> RI;
        constexpr RI it1 = cuda::std::make_reverse_iterator(p);
        constexpr RI it2 = cuda::std::make_reverse_iterator(p+1);
        static_assert( it1 - it2 ==  1, "");
        static_assert( it2 - it1 == -1, "");
    }
#endif

  return 0;
}
